<?php
/**
 *   @copyright Copyright (c) 2018 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 */

class MailChimpV3_Config extends Gpf_Plugins_Config {

    const API_KEY = 'MailChimpV3ApiKey';
    const SECRET_KEY = 'MailChimpV3SecretKey';
    const LIST_ID = 'MailChimpV3ListId';
    const LIST_ID_MERCHANT = 'MailChimpV3ListIdMerchant';
    const ADD_NEW = 'MailChimpV3AddNew';

    protected function initFields() {
        $this->addTextBox($this->_('API key'), self::API_KEY, $this->_('You have to enter a valid API Key so the system could communicate with your MailChimp. Get a key in %s section', '<a target="_blank" href="https://admin.mailchimp.com/account/api">https://admin.mailchimp.com/account/api</a>'));
        $this->addTextBox($this->_('Secret key for webhooks'), self::SECRET_KEY, $this->_('The secret key will be used to identify a valid data sent from your MailChimp webhook. The key can be any string, e.g. MyPass007'));
        $this->addTextBox($this->_('Audience ID'), self::LIST_ID, $this->_('ID of the list/audience you want an affiliate to be added to (updated in). In case you want to use more lists/audiences, separate it with semicolon. %sHow to find my list/audience ID?%s', '<a target="_blank" href="https://mailchimp.com/help/find-your-list-id/">', '</a>'));
        $this->addTextBox($this->_('Audience ID (merchants)'), self::LIST_ID_MERCHANT, $this->_('ID of the list/audience you want a merchant to be added to (updated in). In case you want to use more lists/audiences, separate it with semicolon. %sHow to find my list/audience ID?%s', '<a target="_blank" href="https://mailchimp.com/help/find-your-list-id/">', '</a>'));
        $values = array(
                Gpf::NO => $this->_('Only update existing users'),
                Gpf::YES => $this->_('Add all new affiliates'),
                Pap_Common_Constants::STATUS_APPROVED => $this->_('Add only approved affiliates')
        );
        $this->addRadioBox($this->_('Add affiliates to MailChimp'), self::ADD_NEW, $values, $this->_('Choose if all or only approved affiliates will be added into your MailChimp list.'));
    }

    /**
     * @service plugin_config write
     * @param Gpf_Rpc_Params $params
     * @return Gpf_Rpc_Form
     */
    public function save(Gpf_Rpc_Params $params) {
        $form = new Gpf_Rpc_Form($params);

        if (trim($form->getFieldValue(self::API_KEY)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('API key'));
            $form->setFieldError(self::API_KEY, $errorMessage);
            $form->setErrorMessage($errorMessage);
            return $form;
        }
        if (!strstr($form->getFieldValue(self::API_KEY), '-')) {
            $errorMessage = $this->_('Wrong API key', $this->_('API key'));
            $form->setFieldError(self::API_KEY, $errorMessage);
            $form->setErrorMessage($errorMessage);
            return $form;
        }
        if (trim($form->getFieldValue(self::SECRET_KEY)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('Secret key for webhooks'));
            $form->setFieldError(self::SECRET_KEY, $errorMessage);
            $form->setErrorMessage($errorMessage);
            return $form;
        }
        if (trim($form->getFieldValue(self::LIST_ID)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('List ID'));
            $form->setFieldError(self::LIST_ID, $errorMessage);
            $form->setErrorMessage($errorMessage);
            return $form;
        }

        Gpf_Settings::set(self::API_KEY, trim($form->getFieldValue(self::API_KEY)));
        Gpf_Settings::set(self::SECRET_KEY, trim($form->getFieldValue(self::SECRET_KEY)));
        Gpf_Settings::set(self::LIST_ID, trim($form->getFieldValue(self::LIST_ID)));
        Gpf_Settings::set(self::LIST_ID_MERCHANT, trim($form->getFieldValue(self::LIST_ID_MERCHANT)));
        Gpf_Settings::set(self::ADD_NEW, $form->getFieldValue(self::ADD_NEW));

        $form->setInfoMessage($this->_('Settings saved'));
        return $form;
    }

    /**
     * @service plugin_config read
     * @param Gpf_Rpc_Params $params
     * @return Gpf_Rpc_Form
     */
    public function load(Gpf_Rpc_Params $params) {
        $form = new Gpf_Rpc_Form($params);

        $form->addField(self::API_KEY, Gpf_Settings::get(self::API_KEY));
        $form->addField(self::SECRET_KEY, Gpf_Settings::get(self::SECRET_KEY));
        $form->addField(self::LIST_ID, Gpf_Settings::get(self::LIST_ID));
        $form->addField(self::LIST_ID_MERCHANT, Gpf_Settings::get(self::LIST_ID_MERCHANT));
        $form->addField(self::ADD_NEW, Gpf_Settings::get(self::ADD_NEW));

        return $form;
    }
}
