<?php
/**
 *   @copyright Copyright (c) 2014 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class Infusionsoft_Config extends Gpf_Plugins_Config {

    const API_KEY = 'InfusionsoftAPIKey';
    const SUBDOMAIN = 'InfusionsoftSubdomain';
    const ACCOUNT_ID = 'InfusionsoftAccountId';
    const REGISTER_AFFILIATE = 'InfusionsoftRegisterAffiliate';
    const SAVE_CONTACT_DETAILS = 'InfusionsoftSaveContactDetails';

    protected function initFields() {
        $this->addTextBox($this->_("API Key"), self::API_KEY, $this->_("See the integration method to find out where to get the API Encrypted key"));
        $this->addTextBox($this->_("Account name"), self::SUBDOMAIN, $this->_("Your Infusionsoft account name (subdomain), found in Your Accounts section right after you login to your Infusionsoft."));
        if ($this->isNetworkActive()) {
            $this->addListBox($this->_('Network Account'), self::ACCOUNT_ID, $this->getAccountIds(), $this->_('Choose network account.'));
        }
        $this->addCheckBox($this->_("Register new affiliate with every occured event"), self::REGISTER_AFFILIATE, $this->_('When this checked, with every event new affiliate will be created from credentials that were set in Infusionsoft submit form.'));
        $this->addCheckBox($this->_("Save additional contact details"), self::SAVE_CONTACT_DETAILS, $this->_('When this checked, contact firstname and lastname will be saved to data2 and phone number to data3 transaction fields.'));
    }

    private function getAccountIds() {
        $accountIds = array();
        $selectBuilder = new Gpf_SqlBuilder_SelectBuilder();
        $selectBuilder->select->add(Gpf_Db_Table_Accounts::ID);
        $selectBuilder->select->addDontQuote('IF('.Gpf_Db_Table_Accounts::ID.'=\''.Gpf_Db_Account::DEFAULT_ACCOUNT_ID.'\', CONCAT('.Gpf_Db_Table_Accounts::NAME.',\' ('.$this->_('owner').')\'),'.Gpf_Db_Table_Accounts::NAME.')', 'value');
        $selectBuilder->from->add(Gpf_Db_Table_Accounts::getName());

        if (!Gpf_Session::getAuthUser()->hasPrivilege(Pap_Privileges::ACCOUNT_NAME, Gpf_Privileges::P_READ)) {
            $selectBuilder->where->add(Gpf_Db_Table_Accounts::ID, '=', Gpf_Session::getAuthUser()->getAccountId());
        }
        $selectBuilder->orderBy->add(Gpf_Db_Table_Accounts::NAME);
        $rows = $selectBuilder->getAllRows();
        foreach ($rows as $row) {
            $accountIds[$row->get(Gpf_Db_Table_Accounts::ID)] = $row->get('value');
        }
        return $accountIds;
    }

    private function isNetworkActive() {
        return Gpf_Plugins_Engine::getInstance()->getConfiguration()->isPluginActive('AffiliateNetwork');
    }

    protected function getTabsCount() {
        $tabsCount = 1;
        $settingValueArray = @unserialize(Gpf_Settings::get(self::SUBDOMAIN));
        if (is_array($settingValueArray)) {
            $arrayKeys = array_keys($settingValueArray);
            sort($arrayKeys);
            $tabsCount = end($arrayKeys) + 1;
        }
        return $tabsCount;
    }

    /**
     * @service plugin_config write
     * @param Gpf_Rpc_Params $params
     * @return Gpf_Rpc_Form
     */
    public function save(Gpf_Rpc_Params $params) {
        $form = new Gpf_Rpc_Form($params);
        $formId = $form->getFieldValue(Gpf_View_FormService::ID);

        $this->setSettingForId($formId, self::API_KEY, $form->getFieldValue(self::API_KEY));
        $this->setSettingForId($formId, self::SUBDOMAIN, $form->getFieldValue(self::SUBDOMAIN));
        $this->setSettingForId($formId, self::REGISTER_AFFILIATE, $form->getFieldValue(self::REGISTER_AFFILIATE));
        $this->setSettingForId($formId, self::SAVE_CONTACT_DETAILS, $form->getFieldValue(self::SAVE_CONTACT_DETAILS));
        if ($this->isNetworkActive()) {
            $this->setSettingForId($formId, self::ACCOUNT_ID, $form->getFieldValue(self::ACCOUNT_ID));
        }

        $form->setInfoMessage($this->_('Infusionsoft settings saved'));
        return $form;
    }

    /**
     * @service plugin_config read
     * @param Gpf_Rpc_Params $params
     * @return Gpf_Rpc_Form
     */
    public function load(Gpf_Rpc_Params $params) {
        $form = new Gpf_Rpc_Form($params);
        $formId = $form->getFieldValue(Gpf_View_FormService::ID);
        if (!is_numeric($formId)) {
            throw new Gpf_Exception('ERROR: Unable to load plugin configuration, form ID is not numeric: '. var_export($formId, true));
        }

        if ($this->isNetworkActive()) {
            $form->addField(self::ACCOUNT_ID, self::getSettingForId($formId, self::ACCOUNT_ID, Gpf_Db_Account::DEFAULT_ACCOUNT_ID));
        }
        $form->addField(self::API_KEY, self::getSettingForId($formId, self::API_KEY, ''));
        $form->addField(self::SUBDOMAIN, self::getSettingForId($formId, self::SUBDOMAIN, ''));
        $form->addField(self::REGISTER_AFFILIATE, self::getSettingForId($formId, self::REGISTER_AFFILIATE, Gpf::NO));
        $form->addField(self::SAVE_CONTACT_DETAILS, self::getSettingForId($formId, self::SAVE_CONTACT_DETAILS, Gpf::NO));
        return $form;
    }
}
