<?php
/**
 *   @copyright Copyright (c) 2007 Quality Unit s.r.o.
 *   @author Ladislav Acs
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class FastSpring_Tracker extends Pap_Tracking_CallbackTracker {

    const TYPE_RECURRING = 'subscription.charge.completed';
    const TYPE_SALE = 'order.completed';
    const TYPE_FAILED_SECRET = 'failed';

    private $initialRequest = null;
    private $event = null;

    /**
     * @return FastSpring_Tracker
     */
    public static function getInstance() {
        $tracker = new FastSpring_Tracker();
        $tracker->setTrackerName("FastSpring");
        return $tracker;
    }

    public function getOrderID() {
        if ($this->isRecurringFeatureActive() && $this->getSubscriptionID() != '') {
            return $this->getSubscriptionID();
        }
        return $this->getTransactionID();
    }

    public function getRecurringTotalCost() {
        $recurringTotal = (isset($this->event->total)) ? $this->event->total : null;
        if (isset($this->event->order->subtotalInPayoutCurrency)) {
            $recurringTotal = $this->event->order->subtotalInPayoutCurrency;
        }
        return $recurringTotal;
    }

    public function checkStatus() {
        if ($this->getType() === false || !in_array($this->getType(), array(self::TYPE_RECURRING, self::TYPE_SALE))) {
            $this->debug('Wrong request or webhook event received. Stopping!');
            return false;
        }
        return true;
    }

    public function isRecurring() {
        if ($this->getType() == self::TYPE_RECURRING) {
            $this->debug('Webhook on recurring payment received. Checking if we can create a recurring commission.');
            return true;
        }
        $this->debug('Creating regular commission.');
        return false;
    }

    public function process() {
        $this->debug("------------------- started -------------------");
        $this->readRequestVariables();
        if ($this->getType() == self::TYPE_FAILED_SECRET) {
            return;
        }
        $this->debug("Iterating through all received events.");
        foreach ($this->initialRequest->events as $event) {
            $this->debug("----------- starting tracking of an event -----------");
            $this->readEventDetails($event);
            if($this->checkCookie()) {
                $this->debug("Checking payment status");
                if($this->checkStatus()) {
                    $this->processAndFillAccountIdAndVisitorId($this->getCookie());
                    if ($this->isAffiliateRegisterAllowed()) {
                        $this->registerAffiliate();
                    }
                    if($this->isRecurring()) {
                        if (!$this->isRecurringFeatureActive()) {
                            $this->debug('Stopping, recurring payment can be tracked only with active Recurring Commissions feature.');
                            continue;
                        }
                        $originalCommissionIdForRecurring = $this->getOriginalTransactionForRecurringIfProductChanged();
                        if ($originalCommissionIdForRecurring === false) {
                            $this->debug('Trying to trigger recurring commissions for subscription ID ' . $this->getSubscriptionID());
                            $this->processSubscriptionPayment($this->getSubscriptionID());
                        } else {
                            $this->debug('Subscription product has changed, tracking the commission as a regular Sale.');
                            $recurringCommissionsMain = Pap_Features_RecurringCommissions_Main::getHandlerInstance();
                            $recurringCommissionsMain->setStatusForRecurringCommissionRule($originalCommissionIdForRecurring, Pap_Common_Constants::STATUS_DECLINED);
                            if (isset($this->event->order->id)) {
                                $this->event = $this->event->order;
                                $this->registerRegularCommission();
                            }
                        }
                    } else {
                        $this->registerRegularCommission();
                    }
                    $this->debug("  Processing callback ended");
                } else {
                    $this->debug("Transaction was not in success state, stopping");
                }
            }
        }
        $this->debug("------------------- ended -------------------");
    }

    public function readRequestVariables() {
        if (!$this->checkWebhookAndSecret()) {
            $this->setType(self::TYPE_FAILED_SECRET);
        }
    }
    
    protected function allowUseRecurringCommissionSettings() {
        return true;
    }
    
    private function readEventDetails($event) {
        $this->event = $event;

        if (Gpf_Settings::get(FastSpring_Config::TEST_MODE) !== Gpf::YES && $this->event->live == false) {
            $this->debug('Test mode request received and test mode tracking disabled in plugin configuration. Stopping.');
            $this->setType(false);
            return;
        }

        $this->setType($this->event->type);
        $this->event = $this->event->data;
        if ($this->getType() == self::TYPE_RECURRING) {
            $this->setSubscriptionID($this->returnSubscriptionId($this->event));
        } else {
            $this->setSubscriptionID('');
        }
        if (isset($this->event->tags) && isset($this->event->tags->papCookie)) {
            $this->setCookie($this->event->tags->papCookie);
        } else {
            $this->setCookie('');
        }
    }

    private function registerRegularCommission() {
        $items = $this->event->items;
        $i = 0;
        foreach ($items as $item) {
            $this->setProductVariables($item, $i);
            $this->registerCommission();
            $i++;
        }
    }

    private function getCorrectDataFromEventFor($desiredData, $item) {
        switch ($desiredData) {
            case '0':
                return '';
            case 'customer_email':
            case 'customer_company':
            case 'customer_phone':
                $data = substr($desiredData, strpos($desiredData, '_') + 1);
                return $this->event->customer->$data;
            case 'customer_name':
                return $this->event->customer->first . ' ' . $this->event->customer->last;
            case 'customer_country':
                return $this->event->address->country;
            case 'customer_currency':
                $desiredData = substr($desiredData, strpos($desiredData, '_') + 1);
            case 'reference':
            case 'order':
                return $this->event->$desiredData;
            default:
                return $item->$desiredData;
        }
    }

    private function checkWebhookAndSecret() {
        $response = @file_get_contents('php://input');
        $this->debug(' webhook data: ' . $response);
        $response_object = json_decode($response);
        if ($response == '' || $response_object == false) {
            $this->error('Empty or incorrect response received. Stopping.');
            return false;
        }
        $secret = trim(Gpf_Settings::get(FastSpring_Config::SECRET));
        if ($secret != '') {
            if (!isset($_SERVER['HTTP_X_FS_SIGNATURE']) || $_SERVER['HTTP_X_FS_SIGNATURE'] == '') {
                $this->error('Secret is set in plugin however we didn\'t receive any. Possible fraud, tracking is stopped. If you believe this is a mistake check the Secret settings in your FastSpring webhook configuration and in your FastSpring plugin configuration in Post affiliate Pro.');
                return false;
            }
            $this->debug('Checking webhook secret.');
            $hash = base64_encode(hash_hmac('sha256', $response, $secret, true));
            if ($hash != $_SERVER['HTTP_X_FS_SIGNATURE']) {
                $this->error('Secrets do not match! Possible fraudulent request, tracking is stopped. If you believe this is a mistake check the Secret settings in your FastSpring webhook configuration and in your FastSpring plugin configuration in Post affiliate Pro.');
                return false;
            }
        }
        $this->initialRequest = $response_object;
        return true;
    }

    private function setProductVariables($item, $i) {
        $this->setTotalCost($item->subtotalInPayoutCurrency);
        $this->setCurrency($this->event->payoutCurrency);
        $this->setTransactionID($this->event->order . '(' . $i . ')');
        $this->setSubscriptionID($this->returnSubscriptionId($item));
        $this->setProductID($item->sku);
        if (isset($item->coupon)) {
            $this->setCoupon($item->coupon);
        } else {
            $this->setCoupon('');
        }
        $this->setData1($this->getCorrectDataFromEventFor(Gpf_Settings::get(FastSpring_Config::DATA1), $item));
        $this->setData2($this->getCorrectDataFromEventFor(Gpf_Settings::get(FastSpring_Config::DATA2), $item));
        $this->setData3($this->getCorrectDataFromEventFor(Gpf_Settings::get(FastSpring_Config::DATA3), $item));
        $this->setData4($this->getCorrectDataFromEventFor(Gpf_Settings::get(FastSpring_Config::DATA4), $item));
        $this->setData5($this->getCorrectDataFromEventFor(Gpf_Settings::get(FastSpring_Config::DATA5), $item));
    }
    
    private function returnSubscriptionId($object) {
        if (isset($object->subscription->id)){
            return $object->subscription->id;
        }
        if (isset($object->subscription)) {
            return $object->subscription;
        }
        return '';
    }
    
    private function getOriginalTransactionForRecurringIfProductChanged() {
        if (!isset($this->event->subscription->sku)) {
            return false;
        }
        $recurringSelect = Pap_Features_RecurringCommissions_Main::getRecurringSelect($this->getSubscriptionID());
        $recurringSelect->limit->set(0, 1);
        $recurringSelect->orderBy->add(Pap_Db_Table_RecurringCommissions::LAST_COMMISSION_DATE, false);
        try {
            $record = $recurringSelect->getOneRow();
        } catch (Gpf_DbEngine_NoRowException $e) {
            return false;
        }
        $originalTransactionID = $record->get(Pap_Db_Table_RecurringCommissions::TRANSACTION_ID);
        $transaction = new Pap_Db_Transaction();
        $transaction->setId($originalTransactionID);
        try {
            $transaction->load();
        } catch (Gpf_DbEngine_NoRowException $e) {
            return $originalTransactionID;
        }
        if ($transaction->getProductId() == $this->event->subscription->sku) {
            return false;
        }
        return $originalTransactionID;
    }
    
    private function isRecurringFeatureActive() {
        return Gpf_Plugins_Engine::getInstance()->getConfiguration()->isPluginActive('RecurringCommissions');
    }
}
?>
