<?php
/**
 *   @copyright Copyright (c) 2017 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class Avangate_Tracker extends Pap_Tracking_CallbackTracker {

    const STATUS_COMPLETE = 'COMPLETE';
    const STATUS_REFUND = 'REFUND';
    const STATUS_REVERSED = 'REVERSED';
    const STATUS_CANCELED = 'CANCELED';

    private function getTransactionIdFromOrderId($orderId){
        $transaction = new Pap_Common_Transaction();
        if ($output = $this->findFirstRecordWithData($transaction, Pap_Db_Table_Transactions::ORDER_ID, $orderId)) {
            $this->debug('Parent transaction for refund found by orderId.');
            return $output->getId();
        }

        throw new Gpf_Exception("Parent transaction for order id: $orderId not found.");
    }

    /**
     * @return Avangate_Tracker
     */
    public static function getInstance() {
        $tracker = new Avangate_Tracker();
        $tracker->setTrackerName('Avangate');
        return $tracker;
    }

    protected function refundChargeback() {
        $transaction = new Pap_Common_Transaction();
        $transaction->processRefundChargeback($this->getTransactionIdFromOrderId($this->getOrderID()), Pap_Db_Transaction::TYPE_REFUND, '', $this->getOrderID(), 0, true);
    }

    public function checkStatus() {
        if (Gpf_Settings::get(Avangate_Config::DECLINE_AFFILIATE) == Gpf::YES) {
            $this->declineAffiliate();
        }

        if ($this->getPaymentStatus() == self::STATUS_REFUND) {
            if (Gpf_Settings::get(Avangate_Config::REFUND) != Gpf::YES) {
                $this->debug('Status "'.self::STATUS_REFUND.'" received, but refund handling is not enabled.');
                return false;
            }
            $this->debug('Transaction '.$this->getOrderID().' will be marked as a refund.');
            try {
                $this->refundChargeback();
                $this->debug('Refund complete, ending processing.');
            } catch (Gpf_Exception $e) {
                $this->debug('Error occurred during transaction register:' . $e->getMessage());
            }
            return false;
        }

        // check payment status
        if ($this->getPaymentStatus() != self::STATUS_COMPLETE) {
            $this->debug('Payment status is NOT COMPLETED. Transaction: '.$this->getOrderID().', payer email: '.$this->getEmail().', status: '.$this->getPaymentStatus());
            return false;
        }

        return true;
    }

    private function declineAffiliate() {
        if (($this->getPaymentStatus() == self::STATUS_REFUND) || ($this->getPaymentStatus() == self::STATUS_CANCELED) || ($this->getPaymentStatus() == self::STATUS_REVERSED)) {
            try {
                $affiliate = Pap_Affiliates_User::loadFromUsername($_POST['CUSTOMEREMAIL']);

                if ($affiliate->getStatus() != Pap_Common_Constants::STATUS_APPROVED) {
                    $affiliate->setStatus(Pap_Common_Constants::STATUS_DECLINED);
                    $affiliate->update(array(Gpf_Db_Table_Users::STATUS));
                    $this->debug('Affiliate with username = '.$_POST['CUSTOMEREMAIL'].' has been declined after status '.$this->getPaymentStatus());
                }
            } catch (Gpf_Exception $e) {
                $this->debug('Error occurred during declining the affiliate '.$_POST['CUSTOMEREMAIL'].' [status '.$this->getPaymentStatus().' received]. Exception: '. $e->getMessage());
            }
        }
    }

    private function sendConfirmation(Pap_Tracking_Request $request) {
        $date = $request->getRequestParameter('IPN_DATE');
        $ipnPid = $request->getRequestParameter('IPN_PID');
        $ipnPname = $request->getRequestParameter('IPN_PNAME');
        $string = '';
        if (isset($ipnPid[0])) {
            $string .= strlen($ipnPid[0]) . $ipnPid[0];
        }
        if (isset($ipnPname[0])) {
            $string .= strlen($ipnPname[0]) . $ipnPname[0];
        }
        $string .= strlen($date) . $date;
        $string .= strlen($date) . $date;

        $this->debug('Confirmation hash computed from: "'.$string.'"');
        $hash = '<EPAYMENT>'.$date.'|'.hash_hmac('md5', $string, Gpf_Settings::get(Avangate_Config::API_SECRET_KEY)).'</EPAYMENT>';
        $this->debug('Confirmation hash: "'.$hash.'"');
        echo $hash;
    }

    /**
     *  @return Pap_Tracking_Request
     */
    protected function getRequestObject() {
        $request = Pap_Contexts_Action::getContextInstance()->getRequestObject();
        $this->debug('Data received: "'.print_r($request, true).'"');
        $this->sendConfirmation($request);
        return $request;
    }

    public function readRequestVariables() {
        $request = $this->getRequestObject();
        $this->debug(' Avangate callback: '.print_r($request, true));

        if ($request->getRequestParameter('ORDERSTATUS') == '') {
            $this->debug('Request empty, ending.');
            return false;
        }

        $cookieValue = '';
        if ($request->getRequestParameter('pap_custom') != '') {
            $cookieValue = stripslashes($request->getRequestParameter('pap_custom'));
        }

        if (empty($cookieValue)) {
            $ipnCustmText = $request->getRequestParameter('IPN_CUSTOM_TEXT');
            if (is_array($ipnCustmText)) {
                $customFieldValue = Gpf_Settings::get(Avangate_Config::CUSTOM_FIELD);
                foreach ($ipnCustmText as $key => $value) {
                    if ($customFieldValue == $value) {
                        $ipnCustomValue = $request->getRequestParameter('IPN_CUSTOM_VALUE');
                        $cookieValue = $ipnCustomValue[$key];
                        $this->debug('Cookie value found in response: '.$cookieValue);
                        break;
                    }
                }
            }
        }

        $this->setIpAddress($request->getRequestParameter('IPADDRESS'));

        $this->setCookie($cookieValue);
        $this->setTotalCost($request->getRequestParameter('IPN_TOTALGENERAL'));
        if ($request->getRequestParameter('REFNOEXT') != '') {
            $this->setTransactionID($request->getRequestParameter('REFNOEXT'));
        } else {
            $this->setTransactionID($request->getRequestParameter('ORDERNO'));
        }
        $ipnPcode = $request->getRequestParameter('IPN_PCODE');
        $this->setProductID($ipnPcode[0]);
        $this->setPaymentStatus($request->getRequestParameter('ORDERSTATUS'));
        $this->setEmail($request->getRequestParameter('CUSTOMEREMAIL'));
        $this->setCurrency($request->getRequestParameter('CURRENCY'));
        $this->setData1($request->getRequestParameter('CUSTOMEREMAIL'));
        $this->setData2($request->getRequestParameter('REFNO'));

        $this->readRequestAffiliateVariables($request);

        $this->readAdditionalVariables($request);
    }

    public function readRequestAffiliateVariables(Pap_Tracking_Request $request) {
        $this->setUserFirstName($request->getRequestParameter('FIRSTNAME'));
        $this->setUserLastName($request->getRequestParameter('LASTNAME'));
        $this->setUserEmail($request->getRequestParameter('CUSTOMEREMAIL'));
        $this->setUserCity($request->getRequestParameter('CITY'));
        $this->setUserAddress($request->getRequestParameter('ADDRESS1'));
    }

    public function readAdditionalVariables(Pap_Tracking_Request $request) {
        if ($this->getData1() == '') {
            $this->setData1($request->getRequestParameter('data1'));
        }
        if ($this->getData2() == '') {
            $this->setData2($request->getRequestParameter('data2'));
        }
        if ($this->getData3() == '') {
            $this->setData3($request->getRequestParameter('data3'));
        }
        if ($this->getData4() == '') {
            $this->setData4($request->getRequestParameter('data4'));
        }
        if ($this->getData5() == '') {
            $this->setData5($request->getRequestParameter('data5'));
        }
        if ($this->getCouponCode() == '') {
            $this->setCoupon($request->getRequestParameter('coupon_code'));
        }
        if ($this->getChannelId() == '') {
            $this->setChannelId($request->getRequestParameter('channelId'));
        }
    }

    protected function processAccountIdAndVisitorId(Pap_Tracking_ActionTracker $saleTracker, $cookie) {
        parent::processAccountIdAndVisitorId($saleTracker, $cookie);
        if (Gpf_Settings::get(Avangate_Config::APPROVE_AFFILIATE) == Gpf::YES) {
            $this->debug('Automatic approval of affiliates with sale is enabled');
            try {
                $affiliate = Pap_Affiliates_User::loadFromUsername($this->getData1());

                if ($affiliate->getStatus() == Pap_Common_Constants::STATUS_PENDING) {
                    $affiliate->setStatus(Pap_Common_Constants::STATUS_APPROVED);
                    $affiliate->update();
                }
            } catch (Gpf_Exception $e) {
                $this->debug('Error occurred during affiliate approval: '.$e->getMessage());
            }
        }
    }

    public function getOrderID() {
        return $this->getTransactionID();
    }

    protected function isAffiliateRegisterAllowed() {
        return (Gpf_Settings::get(Avangate_Config::REGISTER_AFFILIATE) == Gpf::YES);
    }

    protected function prepareSales(Pap_Tracking_ActionTracker $saleTracker) {
        if (Gpf_Settings::get(Avangate_Config::PER_PRODUCT_TRCKING) == GPF::YES) {
            $this->prepareSeparateCartItems($saleTracker);
        } else {
            parent::prepareSales($saleTracker);
        }
    }

    private function prepareSeparateCartItems(Pap_Tracking_ActionTracker $saleTracker) {
        $request = $this->getRequestObject();
        $productIds = $request->getRequestParameter('IPN_PCODE');
        $quantities = $request->getRequestParameter('IPN_QTY');
        $prices = $request->getRequestParameter('IPN_PRICE');

        for ($i = 0; $i<= count($prices); $i++) {
            $sale = $saleTracker->createSale();
            $sale->setTotalCost($prices[$i]*$quantities[$i]);
            $sale->setOrderID($this->getOrderID().'('.($i+1).')');
            $sale->setProductID($productIds[$i]);
            $sale->setData1($this->getData1());
            $sale->setData2($this->getData2());
            $sale->setData3($this->getData3());
            $sale->setData4($this->getData4());
            $sale->setData5($this->getData5());
            $sale->setCoupon($this->getCouponCode());
            $sale->setCurrency($this->getCurrency());
            $sale->setChannelId($this->getChannelId());
            if ($this->getStatus()!='') {
                $sale->setStatus($this->getStatus());
            }
            if ($this->getAffiliateID() != '') {
                $sale->setAffiliateID($this->getAffiliateID());
            }
            if  ($this->getCampaignID() != '') {
                $sale->setCampaignID($this->getCampaignID());
            }

            $this->setVisitorAndAccount($saleTracker, $this->getAffiliateID(), $this->getCampaignID(), $this->getCookie());
            $i++;
        }
    }
}
