<?php
/**
 *   @copyright Copyright (c) 2018 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 */

class MailChimpV3_Main extends Gpf_Plugins_Handler {

    const API_DOMAIN = 'https://%s.api.mailchimp.com/3.0/';
    const LIST_URL = 'lists/%s/members';

    /**
     * @var Gpf_Log_Logger
     */
    private $logger;

    public function __construct() {
    }

    /**
     * @return MailChimpV3_Main
     */
    public static function getHandlerInstance() {
        return new MailChimpV3_Main();
    }

    private function getPluginName() {
        return 'MailChimp';
    }

    public function initSettings(Gpf_Settings_Gpf $context) {
        $context->addDbSetting(MailChimpV3_Config::API_KEY, '');
        $context->addDbSetting(MailChimpV3_Config::SECRET_KEY, '');
        $context->addDbSetting(MailChimpV3_Config::LIST_ID, '');
        $context->addDbSetting(MailChimpV3_Config::LIST_ID_MERCHANT, '');
        $context->addDbSetting(MailChimpV3_Config::ADD_NEW, Gpf::NO);
    }

    private function prepareUserInfo(Pap_Common_User $user) {
        if (!empty($_GET['secretKey'])) {
            return; // skip recurrent change when we are processing webhook
        }

        if ($user->getType() == Pap_Application::ROLETYPE_AFFILIATE) {
            $isAffiliate = true;
        } else {
            if (Gpf_Settings::get(MailChimpV3_Config::LIST_ID_MERCHANT) == '') {
                $this->log('end synchronisation with MailChimp, list for merchants is not configured');
                return;
            }
            $isAffiliate = false;
        }

        $mergeFields = array('FNAME'=>$user->getFirstName(),'LNAME'=>$user->getLastName());
        $email = $user->getEmail();
        if ($user->getId() != '') {
            try {
                $userFromDb = Pap_Common_User::getUserById($user->getId());
                if ($userFromDb->getEmail() != $user->getEmail()) {
                    $email = $userFromDb->getEmail();
                }
            } catch (Gpf_Exception $e) {
                $this->log('unable to load user for email: '.$email.', error: '.$e->getMessage(), Gpf_Log::ERROR);
                return;
            }
        }

        $paramsUpdate = $this->prepareListSubscribeData($email, $mergeFields);
        $paramsSubscribe = $paramsUpdate;
        $paramsSubscribe['status'] = 'subscribed';
        // update user
        $this->synchro($user, $paramsUpdate, $paramsSubscribe, $isAffiliate);
    }

    public function updateUserInfo(Pap_Common_User $user) {
        if ($this->isAddAllAffiliates() || $this->isOnlyUserSynchro()) {
            $this->prepareUserInfo($user);
        }
    }

    public function firstTimeApprovedAffiliate(Pap_Common_User $user) {
        if ($this->isAddOnlyOnApprove()) {
            $this->prepareUserInfo($user);
        }
    }

    private function synchro(Pap_Common_User $user, $paramsUpdate, $paramsSubscribe, $isAffiliate = true) {
        $this->log('Starting synchronisation with MailChimp: '.$user->getEmail());
        if (!strstr($this->getApiKey(), '-')) {
            $this->log('wrong Api key, please check plugin configuration', Gpf_Log::WARNING);
            return;
        }
        $url = sprintf(self::API_DOMAIN, $this->getDatacenter()).self::LIST_URL;
        foreach ($this->getListId($isAffiliate) as $id) {
            $urlPost = sprintf($url, trim($id));
            try {
                $result = self::api('POST', $urlPost, $paramsSubscribe);
            } catch (Exception $e) {
                $this->log('user exists in mailchimp: '.$e->getMessage());
                $urlPut = sprintf($url, trim($id)).'/'.md5(strtolower($user->getEmail()));
                try {
                    $result = self::api('PUT', $urlPut, $paramsUpdate);
                } catch (Exception $e) {
                    $this->log('register user error: '.$e->getMessage(), Gpf_Log::ERROR);
                    return false;
                }
            }

            if (isset($result['id'])){
                $this->log('User data synced with MailChimp');
            } else {
                $readableError = $this->getHumanReadable($result);
                $this->log('Update failed! Error '.$readableError, Gpf_Log::ERROR);
            }
        }
    }

    private function getHumanReadable($var) {
        $result = '';
        if (!is_array($var)) {
            return print_r($var, true);
        }
        foreach ($var as $key => $value) {
            $result .= "$key: $value;\n";
        }
        return $result;
    }

    private function prepareListSubscribeData($email, $mergeFields) {
        $params = array();
        $params['email_address'] = $email;
        $params['merge_fields'] = $mergeFields;
        $params['email_type'] = 'html';
        return $params;
    }

    public function processWebhook() {
        // authorize
        if (empty($_GET['secretKey']) || ($_GET['secretKey'] != Gpf_Settings::get(MailChimpV3_Config::SECRET_KEY))) {
            $this->log('webhook: Unauthorized webhook call... Terminating the process.', Gpf_Log::WARNING);
            exit;
        }

        // check type
        if (!isset($_POST['type'])) {
            $this->log('webhook: Missing type, skipping.', Gpf_Log::WARNING);
            exit;
        }
        $type = $_POST['type'] ;
        if (($type != 'profile') && ($type != 'upemail')) {
            $this->log('webhook: Unsupported type ('.$type.'), skipping.', Gpf_Log::WARNING);
            exit;
        }

        // prepare user data
        if ($type == 'upemail') {
            $email = 'old_email';
        } else {
            $email = 'email';
        }
        $this->log('webhook: Trying to update user '.$_POST['data'][$email].'');

        // try to load the user
        try {
            $user = Pap_Affiliates_User::loadFromUsername($_POST['data'][$email]);
        } catch (Gpf_Exception $e) {
            $this->log('webhook: There was an error while loading user: '.$e->getMessage(), Gpf_Log::ERROR);
            exit;
        }

        // update and then save user changes
        if ($type == 'upemail') {
            $user->setEmail($_POST['data']['new_email']);
        } else {
            $user->setFirstName($_POST['data']['merges']['FNAME']);
            $user->setLastName($_POST['data']['merges']['LNAME']);
        }

        try {
            $user->save();
            $this->log('webhook: The user changes were applied successfully.');
        } catch (Gpf_Exception $e) {
            $this->log('webhook: The user changes were NOT applied, error: '.$e->getMessage(), Gpf_Log::ERROR);
            exit;
        }
    }

    private function getApiKey() {
        return Gpf_Settings::get(MailChimpV3_Config::API_KEY);
    }

    private function getDatacenter() {
        $datacenter = explode('-', $this->getApiKey());
        return $datacenter[1];
    }

    /**
     * @param bool $isAffiliate
     * @return array
     */
    private function getListId($isAffiliate = true) {
        if ($isAffiliate) {
            return explode(';', Gpf_Settings::get(MailChimpV3_Config::LIST_ID));
        }
        return explode(';', Gpf_Settings::get(MailChimpV3_Config::LIST_ID_MERCHANT));
    }

    private function addNew() {
        if ($this->isAddAllAffiliates() || $this->isAddOnlyOnApprove()) {
            return true;
        }
        return false;
    }

    private function isOnlyUserSynchro() {
        if (Gpf_Settings::get(MailChimpV3_Config::ADD_NEW) == Gpf::NO) {
            return true;
        }
        return false;
    }

    private function isAddAllAffiliates() {
        if (Gpf_Settings::get(MailChimpV3_Config::ADD_NEW) == Gpf::YES) {
            return true;
        }
        return false;
    }

    private function isAddOnlyOnApprove() {
        if (Gpf_Settings::get(MailChimpV3_Config::ADD_NEW) == Pap_Common_Constants::STATUS_APPROVED) {
            return true;
        }
        return false;
    }

    // ------------------------------------------------------------------
    // ----------------------------- A P I ------------------------------
    // ------------------------------------------------------------------
    public static function api($method, $url, $params = array()) {
        $query = in_array($method, array(
                'GET',
                'DELETE'
        )) ? $params : array();
        $payload = in_array($method, array(
                'POST',
                'PUT'
        )) ? stripslashes(json_encode($params, JSON_UNESCAPED_UNICODE)) : array();
        $request_headers = in_array($method, array(
                'POST',
                'PUT'
        )) ? array(
                'Content-Type: application/json;'
        ) : array();

        $response = self::curlHttpApiRequest($method, $url, $query, $payload, $request_headers);
        $response = json_decode($response, true);

        if (isset($response['title'])) {
            throw new Exception($response['title'].' ('.$response['status'].') ['.$method.': '.$url.']');
        }

        return $response;
    }

    public static function getConnectionURL($apikey) {
    }

    private static function curlHttpApiRequest($method, $url, $query = '', $payload = '', $request_headers = array()) {
        $url = self::curlAppendQuery($url, $query, $method);
        $ch = curl_init($url);
        self::curlSetopts($ch, $method, $payload, $request_headers);
        $response = self::curlExecFollow($ch, 3);
        $errno = curl_errno($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($errno) {
            throw new Exception("Exception $errno: $error");
        }

        return $response;
    }

    private static function curlAppendQuery($url, $query, $method = 'GET') {
        if ($method != 'GET' || empty($query)) {
            return $url;
        }
        if (is_array($query)) {
            return "$url?" . http_build_query($query);
        } else {
            return "$url?$query";
        }
    }

    private static function curlSetopts($ch, $method, $payload, $request_headers) {
        curl_setopt($ch, CURLOPT_HEADER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 3);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_USERPWD, 'apikey:' . Gpf_Settings::get(MailChimpV3_Config::API_KEY));
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if (!empty($request_headers))
            curl_setopt($ch, CURLOPT_HTTPHEADER, $request_headers);

            if ($method != 'GET' && !empty($payload)) {
                if (is_array($payload))
                    $payload = http_build_query($payload);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            }
    }

    private static function curlExecFollow($ch, $maxredirect = null) {
        $mr = $maxredirect === null ? 5 : intval($maxredirect);

        if (ini_get('open_basedir') == '' && in_array(strtolower(ini_get('safe_mode')), array(
                'off',
                '',
                '0'
        ))) {
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $mr > 0);
            curl_setopt($ch, CURLOPT_MAXREDIRS, $mr);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        } else {
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);

            if ($mr > 0) {
                $original_url = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
                $newurl = $original_url;

                $rch = curl_copy_handle($ch);

                curl_setopt($rch, CURLOPT_HEADER, true);
                curl_setopt($rch, CURLOPT_NOBODY, true);
                curl_setopt($rch, CURLOPT_FORBID_REUSE, false);
                do {
                    curl_setopt($rch, CURLOPT_URL, $newurl);
                    $header = curl_exec($rch);
                    if (curl_errno($rch)) {
                        $code = 0;
                    } else {
                        $code = curl_getinfo($rch, CURLINFO_HTTP_CODE);
                        if ($code == 301 || $code == 302) {
                            preg_match('/Location:(.*?)\n/i', $header, $matches);
                            $newurl = trim(array_pop($matches));

                            // if no scheme is present then the new url is a
                            // relative path and thus needs some extra care
                            if (!preg_match("/^https?:/i", $newurl)) {
                                $newurl = $original_url . $newurl;
                            }
                        } else {
                            $code = 0;
                        }
                    }
                } while ($code && --$mr);

                curl_close($rch);

                if (!$mr) {
                    if ($maxredirect === null) {
                        trigger_error('Too many redirects.', E_USER_WARNING);
                        Gpf_Log::error('MailChimp API: Too many redirects.');
                    } else {
                        $maxredirect = 0;
                    }

                    return false;
                }
                curl_setopt($ch, CURLOPT_URL, $newurl);
            }
        }
        return curl_exec($ch);
    }

    /**
     * @return Gpf_Log_Logger
     */
    private function getLogger() {
        if ($this->logger === null) {
            $this->logger = Pap_Logger::create(Pap_Common_Constants::TYPE_SIGNUP);
        }
        return $this->logger;
    }

    private function log($message, $logLevel = Gpf_Log::DEBUG) {
        $this->getLogger()->log($this->getPluginName().': '.$message, $logLevel);
    }
}
