<?php
/**
 *   @copyright Copyright (c) 2007 Quality Unit s.r.o.
 *   @author Viktor Zeman
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro
 */
class GetResponseSignup_JsonRPCClient extends GetResponseSignup_Client
{
    protected $is_notification = false, $is_debug = false;

    // http errors - more can be found at
    // http://en.wikipedia.org/wiki/List_of_HTTP_status_codes
    public $http_errors = array
    (
        400 => '400 Bad Request',
        500 => '500 Internal Server Error'
    );

    protected function getPluginName() {
        return 'GetResponseSignup(old_360)';
    }

    /**
     * Takes the connection parameter and checks for extentions
     *
     * @param string $url - url name like http://example.com/
     * @return void
     */
    public function __construct($url, $apiKey = null) {
        parent::__construct($url, $apiKey);
        $validateParams = array
        (
            false === extension_loaded('curl') => 'The curl extension must be loaded for using this class !',
            false === extension_loaded('json') => 'The json extension must be loaded for using this class !'
        );
        $this->checkForErrors( $validateParams );
    }

    /**
     * Set debug mode
     *
     * @param boolean $is_debug
     * @return void
     */
    public function setDebug( $is_debug )
    {
        $this->is_debug = !empty($is_debug);
    }

    /**
     * Set request to be a notification
     *
     * @param boolean $is_notification
     * @return void
     */
    public function setNotification( $is_notification  )
    {
        $this->is_is_notification = !empty($is_notification);
    }

    private function callFunction($functionName, $params) {
        $this->log('callFunction ' . $functionName . ', params: ' . print_r($params, true), Gpf_Log::INFO);
        try {
            $result = $this->$functionName($this->apiKey, $params);
            $this->log('callFunction result: ' . print_r($result, true), Gpf_Log::INFO);
            return $result;
        } catch (Exception $e) {
            $this->log('Failed to call function to GetResponse "'.$functionName.'" with params: "'.var_export($params, true).'", error: ' . $e->getMessage(), Gpf_Log::ERROR);
        }
    }

    /**
     * Performs a request and gets the results
     *
     * @param string $method - A String containing the name of the method to be invoked.
     * @param array $params - An Array of objects to pass as arguments to the method.
     * @return array
     */
    public function __call( $method, $params )
    {
        static $counter;

        // check if given params are correct
        $validateParams = array
        (
             false === is_scalar($method) => 'Method name has no scalar value',
             false === is_array($params) => 'Params must be given as array'
        );
        $this->checkForErrors( $validateParams );

        // if this is_notification - JSON-RPC specification point 1.3
        $requestId = true === $this->is_notification ? null : ++$counter;

        // Request (method invocation) - JSON-RPC specification point 1.1
        $request = json_encode( array ( 'method' => $method, 'params' => array_values($params), 'id' => $requestId ) );

        // if is_debug mode is true then add request to is_debug
        $this->debug( 'Request: ' . $request . "\r\n", false );

        $response = $this->getResponse( $request );

        // if is_debug mode is true then add response to is_debug and display it
        $this->debug( 'Response: ' . $response . "\r\n", true );

        // decode and create array ( can be object, just set to false )
        $response = json_decode( utf8_encode($response), true );

        // if this was just is_notification
        if ( true === $this->is_notification )
        {
            return true;
        }

        // check if response is correct
        $validateParams = array
        (
            !is_null($response['error']) => 'Request have return error: ' . print_r($response['error'], true),
            $response['id'] != $requestId => 'Request id: '.$requestId.'is different from Response id: ' . $response['id'],

        );
        $this->checkForErrors( $validateParams );

        return $response['result'];
    }

    /**
     *     When the method invocation completes, the service must reply with a response.
     *     The response is a single object serialized using JSON
     *
     * @param string $request
     * @return string
     */
    protected function & getResponse( & $request )
    {
        // do the actual connection
        $ch = curl_init();
        // set URL
        curl_setopt($ch, CURLOPT_URL, $this->url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $request);
        curl_setopt($ch, CURLOPT_HEADER, 'Content-type: application/json;');
        curl_setopt($ch, CURLOPT_ENCODING, 'gzip,deflate');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 5);
        // send the request
        $response = curl_exec($ch);
        // check http status code
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if ( isset($this->http_errors[$http_code])  )
        {
            throw new Exception('Response Http Error - ' . $this->http_errors[$http_code] );
        }
        // check for curl error
        if ( 0 < curl_errno($ch) )
        {
            throw new Exception('Unable to connect to '.$this->url . ' Error: ' . curl_error($ch) );
        }
        // close the connection
        curl_close($ch);
        return $response;
    }

    /**
     * Check for errors
     *
     * @param array $validateArray
     * @return void
     */
    protected function checkForErrors( & $validateArray )
    {
        foreach ( $validateArray as $test => $error )
        {
            if ( $test )
            {
                throw new Exception( $error );
            }
        }
    }

    /**
     * For is_debug and performance stats
     *
     * @param string $add
     * @param boolean $show
     * @return void
     */
    protected function debug( $add, $show = false )
    {
        static $debug, $startTime;
        // is_debug off return
        if ( false === $this->is_debug )
        {
            return;
        }
        // add
        $debug .= $add;
        // get starttime
        $startTime = empty($startTime) ? array_sum(explode(' ', microtime())) : $startTime;
        if ( true === $show and !empty($debug)  )
        {
            // get endtime
            $endTime = array_sum(explode(' ', microtime()));
            // performance summary
            $debug .= 'Request time: ' . round($endTime - $startTime, 3) . ' s Memory usage: ' . round(memory_get_usage() / 1024) . " kb\r\n";
            echo nl2br($debug);
            // send output imidiately
            flush();
            // clean static
            $debug = $startTime = null;
        }
    }

    public function changeEmail(Pap_Common_User $user, $oldEmail) {
        $this->deleteContact($user, $oldEmail);
        $this->signup($user);
        $this->storeContactEmail($user);
    }

    public function deleteContact($email) {
        $contactId = $this->getContactId($email);
        if ($contactId != null) {
            $this->callFunction('delete_contact', array('contact' => $contactId));
        }
    }

    private function getContactId($email) {
        $result = $this->callFunction('get_contacts', array('email' => array('EQUALS' => $email)));
        if (!is_array($result)) {
            return null;
        }
        $resultKeys = array_keys($result);
        return array_pop($resultKeys);
    }

    public function signup(Pap_Common_User $user, $oldEmail = null) {
        $campaignId = $this->resolveCampaignId();

        $cycleDay = Gpf_Settings::get(GetResponseSignup_Config::CYCLE_DAY);
        if ($cycleDay != null && $cycleDay != '') {
            $result = $this->callFunction('add_contact', array (
                    'campaign'  => $campaignId,
                    'name'      => $user->getFirstName() . ' ' . $user->getLastName(),
                    'email'     => $user->getEmail(),
                    'cycle_day' => $cycleDay,
                    'customs'   => $this->getCustomFields($user)));
        } else {
            $result = $this->callFunction('add_contact', array (
                    'campaign'  => $campaignId,
                    'name'      => $user->getFirstName() . ' ' . $user->getLastName(),
                    'email'     => $user->getEmail(),
                    'customs'   => $this->getCustomFields($user)));
        }
        $this->storeContactEmail($user);
    }

    private function resolveCampaignId() {
        $this->log('Trying to resolve campaign ' . Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_CAMPAIGN_NAME), Gpf_Log::INFO);
        $result = $this->callFunction('get_campaigns', array (
                'name' => array ( 'EQUALS' => Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_CAMPAIGN_NAME) )
        ));
        return @array_pop(array_keys($result));
    }

    private function getCustomFields(Pap_Common_User $user) {
        $customFields = explode(',', Gpf_Settings::get(GetResponseSignup_Config::CUSTOM_DATA_FIELDS));

        $select = new Gpf_SqlBuilder_SelectBuilder();
        $select->select->add(Gpf_Db_Table_FormFields::NAME);
        $select->select->add(Gpf_Db_Table_FormFields::CODE);
        $select->from->add(Gpf_Db_Table_FormFields::getName());

        array_walk($customFields, function(&$val) {$val = trim(strtolower($val));});
        $select->where->add(Gpf_Db_Table_FormFields::CODE, 'IN', $customFields);
        $select->where->add(Gpf_Db_Table_FormFields::FORMID, '=', Pap_Merchants_Config_AffiliateFormDefinition::FORMID);
        $customs = array();
        foreach($select->getAllRows() as $row) {
            $name = str_replace(' ', '_', Gpf_Lang::_localizeRuntime($row->get(Gpf_Db_Table_FormFields::NAME)));
            $content = $user->getData(str_replace('data', '', $row->get(Gpf_Db_Table_FormFields::CODE)));
            $customs[] = (object) array(
                    'name' => Gpf_Common_String::utf8_substr($name, 0, self::MAX_GET_RESPONSE_FIELD_NAME),
                    'content' => Gpf_Common_String::utf8_substr($content, 0, self::MAX_GET_RESPONSE_FIELD_CONTENT)
            );
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::INCLUDE_REFERRALID) == Gpf::YES) {
            $customs[] = (object) array(
                    'name' => 'referralid',
                    'content' => $user->getRefId()
            );
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::INCLUDE_PASSWORD) == Gpf::YES) {
            $customs[] = (object) array(
                    'name' => 'affpassword',
                    'content' => $user->getPassword()
            );
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::INCLUDE_PARENT_USER_ID) == Gpf::YES && $user->getParentUserId() !== ''){
            $customs[] = (object) array(
                    'name' => 'parentuserid',
                    'content' => $user->getParentUserId()
            );
        }
        return $customs;
    }
}
?>
