<?php
/**
 *   @copyright Copyright (c) 2018 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class BigCommerceAPIv3_Tracker extends Pap_Tracking_CallbackTracker {

    private $products = array();
    private $pluginConfig = 0;

    /**
     * @return BigCommerceAPIv3_Tracker
     */
    public static function getInstance() {
        $tracker = new BigCommerceAPIv3_Tracker();
        $tracker->setTrackerName('BigCommerceAPIv3');
        return $tracker;
    }

    public function checkStatus() {
        if ($this->getPaymentStatus() == 'OK') {
            return true;
        }
        if ($this->getPaymentStatus() == 'STATUS_CHANGE') {
            $this->changeCommissionStatus();
            return false;
        }
        $this->debug('Invalid status received: ' . $this->getPaymentStatus());
        return false;
    }

    /**
     * @param string $orderid
     * @param string $status
     * @return Gpf_SqlBuilder_SelectIterator
     */
    private function getTransactionsListSelect($orderid, $status) {
        $select = new Gpf_SqlBuilder_SelectBuilder();
        $select->select->addAll(Pap_Db_Table_Transactions::getInstance(), 't');
        $select->from->add(Pap_Db_Table_Transactions::getName(), 't');
        $select->where->add(Pap_Db_Table_Transactions::R_TYPE, '=', Pap_Common_Constants::TYPE_SALE);
        if ($status == Pap_Common_Constants::STATUS_APPROVED) { // approve only pending commissions
            $select->where->add(Pap_Db_Table_Transactions::R_STATUS, '=', Pap_Common_Constants::STATUS_PENDING);
        }
        if ($status == Pap_Common_Constants::STATUS_DECLINED) { // return all commissions that are not declined
            $select->where->add(Pap_Db_Table_Transactions::R_STATUS, '!=', Pap_Common_Constants::STATUS_DECLINED);
        }
        $whereCondition = new Gpf_SqlBuilder_CompoundWhereCondition();
        $whereCondition->add(Pap_Db_Table_Transactions::ORDER_ID, '=', $orderid);
        $whereCondition->add(Pap_Db_Table_Transactions::ORDER_ID, 'LIKE', $orderid . '(%', 'OR');
        $select->where->addCondition($whereCondition);
        $select->orderBy->add(Pap_Db_Table_Transactions::ORDER_ID);
        $select->orderBy->add(Pap_Db_Table_Transactions::TIER);
        return $select->getAllRowsIterator();
    }

    protected function computeTotalCost(array $product) {
        $discountAmount = 0;
        $discounts = $product['applied_discounts'];
        foreach ($discounts as $discountObject) {
            $discountAmount += $discountObject['amount'];
        }
        return $product['total_ex_tax'] - $discountAmount;
    }

    private function changeCommissionStatus() {
        /*
         *    0 = Incomplete
         *    1 = Pending
         *    2 = Shipped
         *    3 = Partially Shipped
         *    4 = Refunded
         *    5 = Cancelled
         *    6 = Declined
         *    7 = Awaiting Payment
         *    8 = Awaiting Pickup
         *    9 = Awaiting Shipment
         *    10 = Completed
         *    11 = Awaiting Fulfillment
         *    12 = Manual Verification Required
         *    13 = Disputed
         *    14 = Partially Refunded
         */
        $orderID = $this->getTransactionID();
        $statusID = $this->getStatus();

        // load orders and change it's status
        if (in_array($statusID, array(0, 4, 5, 6, 13, 14))) {
            $status = Pap_Common_Constants::STATUS_DECLINED;
        } elseif ($statusID == '10') {
            $status = Pap_Common_Constants::STATUS_APPROVED;
        } else {
            // unknown state, this webhook was not registered!
            $this->debug('ignoring status "'.$statusID.'"');
            return false;
        }
        $this->setStatus($status);

        // read data
        $this->debug('Trying to change status to: ' . $status . ', orderid: ' . $orderID);

        $foundTransactions = false;
        $transactionList = $this->getTransactionsListSelect($orderID, $status);
        foreach ($transactionList as $transactionRecord) {
            $foundTransactions = true;
            $transaction = new Pap_Common_Transaction();
            $transaction->fillFromRecord($transactionRecord);

            $this->setLogGroup($transactionRecord->get(Pap_Db_Table_Transactions::LOGGROUPID));

            // change status
            try {
                $transaction->setStatus($status);
                $transaction->update(array(
                        Pap_Db_Table_Transactions::R_STATUS
                ));
                $this->debug('Status has been changed.');
            } catch (Gpf_DbEngine_Row_ConstraintException $e) {
                $this->error('Error changing status of order ' . $orderID . '; error message: ' . $e->getMessage());
            }
        }

        if (!$foundTransactions) {
            $unprocessedTransaction = new Pap_Merchants_Transaction_TransactionsForm();
            try {
                if ($unprocessedTransaction->changeStatusPerOrderIdNoRpc($orderID, $status, '')) {
                    $this->debug('Status of order "' . $orderID . '" has been changed.');
                } else {
                    $this->error('No transaction found for status changing.');
                }
            } catch (Pap_Tasks_TaskCreatedException $e) {
                $this->error('Error, during changing status: '.$e->getMessage());
            }
        }

        return true;
    }

    /**
     *  @return Pap_Tracking_Request
     */
    protected function getRequestObject() {
        $request = Pap_Contexts_Action::getContextInstance()->getRequestObject();
        $this->debug('Request contains: ' . print_r($request->toString(), true));
        $input = file_get_contents('php://input');
        $this->debug('Input contains: ' . print_r($input, true));
        if ($request == null || $request->toString() == '') {
            $this->debug('Handling a webhook notification');
            $inputRequest = json_decode($input, true);
            if ($inputRequest != null) {
                return $inputRequest;
            }
        }
        return $request;
    }

    public function readRequestVariables() {
        $request = $this->getRequestObject();
        if (! $request instanceof Pap_Tracking_Request) {
            //it is a webhook
            $this->setTransactionID($request['data']['id']);
            $this->setStatus($request['data']['status']['new_status_id']);
            $this->setPaymentStatus('STATUS_CHANGE');
            return false;
        }
        $this->initPluginConfig($request->getRequestParameter('configId'));
        if (BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::API_PATH) == '' ||
            BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::API_TOKEN) == '' ||
            BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::API_USERNAME) == '') {
            $this->error('Please configure your plugin first! The API data are missing!');
            $this->setPaymentStatus('FAILED');
            return false;
        }
        $orderId = $request->getRequestParameter('orderId');
        $cookie = $request->getRequestParameter('visitorId');
        $email = $request->getRequestParameter('email');
        $this->debug("Received order with ID: '$orderId', the cookie value is: '$cookie' and customer's email is: '$email''");

        if (empty($orderId)) {
            $this->setPaymentStatus('EMPTY');
            return false;
        }

        $this->setCookie($cookie);
        $this->setTransactionID($orderId);

        $connection = array(
                'url' => BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::API_PATH),
                'clientId' => BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::API_USERNAME),
                'token' => BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::API_TOKEN)
        );

        // load order details
        $response = BigCommerceAPIv3_Main::api('GET', 'v2/orders/' . $orderId, $connection, array());
        $this->debug('Order data received: ' . var_export($response, true));
        if (!is_array($response)) {
            $this->error('API error, Unable to load order details for "'.$orderId.'", API response: '.var_export($response, true));
            return;
        }
        if (array_key_exists('status', $response) && $response['status'] >= 400) {
            $this->error('API error, Unable to load order details for "'.$orderId.'": ' . print_r($response, true));
            return;
        }

        $this->setPaymentStatus('OK');

        $this->setCurrency($response['currency_code']);
        $this->setEmail($response['billing_address']['email']);
        $this->setUserFirstName($response['billing_address']['first_name']);
        $this->setUserLastName($response['billing_address']['last_name']);
        $this->setUserAddress($response['billing_address']['country']);
        $this->setTotalCost($response['total_ex_tax']);

        if (BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::COUPON_TRACKING) == Gpf::YES &&
                isset($response['coupons']) &&
                isset($response['coupons']['resource']) &&
                ($response['coupons']['resource'] != '')) {
            $coupons = BigCommerceAPIv3_Main::api('GET', 'v2'.$response['coupons']['resource'], $connection);
            if (is_array($coupons)) {
                $this->setCoupon($coupons[0]['code']);
            } else {
                $this->error('API error, Unable to load coupons for "'.$orderId.'"(' .$response['coupons']['resource']. '), API response: '.var_export($coupons, true));
            }
        }

        // load product details
        $products = BigCommerceAPIv3_Main::api('GET', 'v2/orders/' . $orderId . '/products', $connection);
        $this->debug('Order products data received: ' . print_r($products, true));
        if (!is_array($products)) {
            $this->error('API error, Unable to load products for order "'.$orderId.'", API response: '.var_export($products, true));
            return;
        }
        if (array_key_exists('status', $products) && $products['status'] >= 400) {
            $this->error('API error, Unable to load products for order "'.$orderId.'": ' . print_r($products, true));
            return;
        }
        if ($this->isPerProductTracking()) {
            $this->products = $products;
        }

        for ($i = 1; $i <= 5; $i++) {
            if (BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, constant('BigCommerceAPIv3_Config::DATA'.$i)) !== 0) {
                $this->setData($i, $this->translateData(BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, constant('BigCommerceAPIv3_Config::DATA'.$i)), $products[0]));
            }
        }
    }

    protected function prepareSales(Pap_Tracking_ActionTracker $saleTracker) {
        if (!$this->isPerProductTracking()) {
            parent::prepareSales($saleTracker); // per order tracking
            return;
        }

        $i = 1;
        foreach ($this->products as $product) {
            $sale = $saleTracker->createSale();
            $sale->setTotalCost($this->computeTotalCost($product));
            $sale->setOrderID($product['order_id'] . '(' . $i . ')');

            $sale->setProductID($product['product_id']);
            if (BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::PER_PRODUCT) == 'sku' && ($product['sku'] != '')) {
                $sale->setProductID($product['sku']);
            }

            $sale->setData1($this->translateData(BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::DATA1), $product));
            $sale->setData2($this->translateData(BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::DATA2), $product));
            $sale->setData3($this->translateData(BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::DATA3), $product));
            $sale->setData4($this->translateData(BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::DATA4), $product));
            $sale->setData5($this->translateData(BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::DATA5), $product));

            $sale->setCoupon($this->getCouponCode());
            $sale->setCurrency($this->getCurrency());

            $this->setVisitorAndAccount($saleTracker, $this->getAffiliateID(), $this->getCampaignID(), $this->getCookie());
            $i++;
        }
    }

    private function isPerProductTracking() {
        return BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::PER_PRODUCT) !== '0';
    }

    private function translateData($replace, $product) {
        switch ($replace) {
            case 'product_id':
                if ($product != '') {
                    return $product['product_id'];
                } else {
                    if (BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::PER_PRODUCT) == 'id'){
                        return $this->getProductID();
                    }
                    return '';
                }
                break;
            case 'sku':
                if ($product != '') {
                    return $product['sku'];
                } else {
                    if (BigCommerceAPIv3_Config::getSettingForId($this->pluginConfig, BigCommerceAPIv3_Config::PER_PRODUCT) == 'sku') {
                        return $this->getProductID();
                    }
                    return '';
                }
                break;
            case 'product_name':
                if ($product != '') {
                    return $product['name'];
                } else {
                    return '';
                }
                break;
            case 'base_price':
                if ($product != '') {
                    return $product['base_price'];
                } else {
                    return '';
                }
                break;
            case 'quantity':
                if ($product != '') {
                    return $product['quantity'];
                } else {
                    return '';
                }
                break;
            case 'customer_email':
                return $this->getEmail();
                break;
            case 'customer_name':
                return $this->getUserFirstName().' '.$this->getUserLastName();
                break;
            case 'customer_country':
                return $this->getUserAddress();
                break;
        }
    }

    public function getOrderID() {
        return $this->getTransactionID();
    }

    private function initPluginConfig($configIdentifier) {
        $this->pluginConfig = 0;
        $configIdentifier = trim($configIdentifier);
        if ($configIdentifier != '') {
            $settingValueArray = @unserialize(Gpf_Settings::get(BigCommerceAPIv3_Config::CONFIG_IDENTIFIER));
            if ($settingValueArray !== false && is_array($settingValueArray)) {
                $configKey = array_search($configIdentifier, $settingValueArray);
                if ($configKey !== false) {
                    $this->pluginConfig = $configKey;
                }
            }
        }
    }
}
