<?php
/**
 *   @copyright Copyright (c) 2017 Quality Unit s.r.o.
 *   @author Lubomir Nikodem
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliate
 */
class GetResponseSignup_V3Client extends GetResponseSignup_Client {

    const MAX_GET_RESPONSE_FIELD_CONTENT = 128;

    private $customFields = null;

    private $is360V3Api = false;
    private $getResponse360Domain = '';

    public function __construct($url, $apiKey = null, $is360V3Api = false, $getResponse360Domain = '') {
        parent::__construct($url, $apiKey);

        $this->is360V3Api = $is360V3Api;
        $this->getResponse360Domain = $getResponse360Domain;
    }

    protected function execute(Gpf_Net_Http_Request $request, Pap_Common_User $user = null) {
        try {
            $client = new Gpf_Net_Http_Client();
            $response = $client->execute($request, (new Gpf_Net_Http_SafeUrl_Options())->setFollowLocationLimit(3)->enableFollowLocation());
            $responseBody = Gpf_Rpc_Json::decodeStatic($response->getBody(), true);
            if ($response->getResponseCode() >= 400 && $response->getResponseCode() < 600) {
                if ($response->getResponseCode() == 409 && $user != null) {
                    $this->storeContactEmail($user);
                }
                throw new Gpf_Exception($responseBody['message'] . '. Response: '. print_r($responseBody, true));
            }
            $this->log('callFunction result: ' . print_r($responseBody, true), Gpf_Log::INFO);
            return $responseBody;
        } catch (Gpf_Exception $e) {
            $this->log('Failed to execute request to GetResponse "'.$request->getUrl().'" request body: "'.var_export($request->getBody(), true).'", error: ' . $e->getMessage(), Gpf_Log::ERROR);
        }
        return false;
    }

    public function deleteContact($email) {
        $contactId = $this->getContactId($email);
        if ($contactId != null) {
            $request = $this->createRequest();
            $request->setUrl($this->url . '/contacts/' . $contactId);
            $request->setMethod('DELETE');
            $this->execute($request);
        }
    }

    public function getContact($email) {
        $request = $this->createRequest();
        $queryParams = array(
                'query' => array(
                        'email' => $email
                ),
                'fields' => 'email,campaign',
        );
        $request->setUrl($this->url . '/contacts?' . http_build_query($queryParams));
        $request->setMethod('GET');
        $result = $this->execute($request);
        if (empty($result) || !is_array($result)) {
            return null;
        }
        return $result[0];
    }

    public function getContactId($email) {
        $contact = $this->getContact($email);
        if ($contact == null) {
            return null;
        }
        return $contact['contactId'];
    }

    public function changeEmail(Pap_Common_User $user, $oldEmail) {
        $this->signup($user, $oldEmail);
    }

    protected function resolveCampaignId() {
        $this->log('Trying to resolve campaign ' . $this->getCampaignName(), Gpf_Log::INFO);
        $request = $this->createRequest();
        $queryParams = array(
                'query' => array(
                        'name' => $this->getCampaignName()
                )
        );
        $request->setUrl($this->url . '/campaigns?' . http_build_query($queryParams));
        $request->setMethod('GET');
        $result = $this->execute($request);
        if (empty($result) || !is_array($result)) {
            return null;
        }
        return $result[0]['campaignId'];
    }

    protected function getCampaignName() {
        return Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_CAMPAIGN_NAME);
    }

    public function signup(Pap_Common_User $user, $oldEmail = null) {
        $campaignId = $this->resolveCampaignId();
        $contactId = '';
        if ($oldEmail != null) {
            $contactId = '/' . $this->getContactId($oldEmail);
        }

        $params = (object) array(
                'campaign' => (object) array(
                        'campaignId' => $campaignId
                ),
                'name' => $user->getFirstName() . ' ' . $user->getLastName(),
                'email' => $user->getEmail(),
                'customFieldValues' => $this->getCustomFields($user)
        );
        $cycleDay = Gpf_Settings::get(GetResponseSignup_Config::CYCLE_DAY);
        if ($cycleDay != null && $cycleDay != '') {
            $params->dayOfCycle = $cycleDay;
        }
        $request = $this->createRequest();
        $request->setUrl($this->url . '/contacts' . $contactId);
        $request->setMethod('POST');
        $request->setBody(Gpf_Rpc_Json::encodeStatic($params));
        $result = $this->execute($request, $user);
        $this->log('Affiliate added', Gpf_Log::INFO);

        if ($result != false) {
            $this->storeContactEmail($user);
        }
    }

    private function getCustomFields(Pap_Common_User $user) {
        $customFields = explode(',', Gpf_Settings::get(GetResponseSignup_Config::CUSTOM_DATA_FIELDS));

        $select = new Gpf_SqlBuilder_SelectBuilder();
        $select->select->add(Gpf_Db_Table_FormFields::NAME);
        $select->select->add(Gpf_Db_Table_FormFields::CODE);
        $select->from->add(Gpf_Db_Table_FormFields::getName());

        array_walk($customFields, function(&$val) {$val = trim(strtolower($val));});
        $select->where->add(Gpf_Db_Table_FormFields::CODE, 'IN', $customFields);
        $select->where->add(Gpf_Db_Table_FormFields::FORMID, '=', Pap_Merchants_Config_AffiliateFormDefinition::FORMID);
        $customs = array();
        foreach ($select->getAllRows() as $row) {
            $name = str_replace(' ', '_', Gpf_Lang::_localizeRuntime($row->get(Gpf_Db_Table_FormFields::NAME)));
            $content = $user->getData(str_replace('data', '', $row->get(Gpf_Db_Table_FormFields::CODE)));
            $customField = $this->getCustomField(
                    Gpf_Common_String::utf8_substr($name, 0, self::MAX_GET_RESPONSE_FIELD_NAME),
                    Gpf_Common_String::utf8_substr($content, 0, self::MAX_GET_RESPONSE_FIELD_CONTENT)
            );
            if ($customField) {
                $customs[] = $customField;
            }
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::INCLUDE_REFERRALID) == Gpf::YES) {
            $customField = $this->getCustomField('referralid', $user->getRefId());
            if ($customField) {
                $customs[] = $customField;
            }
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::INCLUDE_PASSWORD) == Gpf::YES) {
            $customField = $this->getCustomField('affpassword', $user->getPassword());
            if ($customField) {
                 $customs[] = $customField;
            }
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::INCLUDE_PARENT_USER_ID) == Gpf::YES && $user->getParentUserId() !== '') {
            $customField = $this->getCustomField('parentuserid', $user->getParentUserId());
            if ($customField) {
                $customs[] = $customField;
            }
        }
        return $customs;
    }

    private function getCustomField($name, $value) {
        $customFieldId = $this->getCustomFieldId($name);
        if ($customFieldId) {
            return (object) array(
                    'customFieldId' => $customFieldId,
                    'value' => array(
                            $value
                    )
            );
        }
        return false;
    }

    private function getCustomFieldId($fieldName) {
        $customFields = $this->getFields();
        if ($customFields == null) {
            $this->log('Unable to load custom fields', Gpf_Log::ERROR);
            return false;
        }
        foreach ($customFields as $customField) {
            if ($customField['name'] == $fieldName) {
                return $customField['customFieldId'];
            }
        }
        $customField = $this->createCustomField($fieldName);
        if (!$customField) {
            return false;
        }
        return $customField['customFieldId'];
    }

    private function createCustomField($fieldName) {
        $params = (object) array(
                'name' => $fieldName,
                'type' => 'text',
                'hidden' => 'false',
                'values' => (object) array()
        );
        $request = $this->createRequest();
        $request->setUrl($this->url . '/custom-fields');
        $request->setMethod('POST');
        $request->setBody(Gpf_Rpc_Json::encodeStatic($params));
        $result = $this->execute($request);
        if ($result != false) {
            $this->customFields[] = $result;
        }
        return $result;
    }

    private function getFields() {
        if ($this->customFields != null) {
            return $this->customFields;
        }

        $request = $this->createRequest();
        $request->setUrl($this->url . '/custom-fields');
        $request->setMethod('GET');
        $result = $this->execute($request);
        if (empty($result) || !is_array($result)) {
            return null;
        }
        $this->customFields = $result;
        return $this->customFields;
    }

    /**
     *
     * @return Gpf_Net_Http_Request
     */
    protected function createRequest() {
        $request = new Gpf_Net_Http_Request();
        $request->setHeader('Content-type', 'application/json');
        $request->setHeader('X-Auth-Token', 'api-key ' . $this->apiKey);
        if ($this->is360V3Api) {
            $request->setHeader('X-Domain', $this->getResponse360Domain);
        }
        return $request;
    }
}
