<?php
/**
 *   @copyright Copyright (c) 2007 Quality Unit s.r.o.
 *   @author Viktor Zeman
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro
 */
class GetResponseSignup_Main extends Gpf_Plugins_Handler {

    /**
     *
     * @var GetResponseSignup_Client
     */
    private $client;
    private $apiKey;

    private $isSubscribed;

    /**
     * @var Gpf_Log_Logger
     */
    private $logger;

    const GET_RESPONSE_CONTACT_EMAIL = 'getResponseContactEmail';

    protected function getPluginName() {
        return 'GetResponseSignup';
    }

    public function initSettings(Gpf_Settings_Gpf $context) {
        $context->addDbSetting(GetResponseSignup_Config::GETRESPONSE_API_KEY, '');
        $context->addDbSetting(GetResponseSignup_Config::GETRESPONSE_CAMPAIGN_NAME, '');
        $context->addDbSetting(GetResponseSignup_Config::CUSTOM_DATA_FIELDS, '');
        $context->addDbSetting(GetResponseSignup_Config::INCLUDE_REFERRALID, Gpf::NO);
        $context->addDbSetting(GetResponseSignup_Config::INCLUDE_PASSWORD, Gpf::NO);
        $context->addDbSetting(GetResponseSignup_Config::INCLUDE_PARENT_USER_ID, Gpf::NO);
        $context->addDbSetting(GetResponseSignup_Config::CYCLE_DAY, '');
        $context->addDbSetting(GetResponseSignup_Config::GETRESPONSE_360, Gpf::NO);
        $context->addDbSetting(GetResponseSignup_Config::GETRESPONSE_360_API_URL, '');
        $context->addDbSetting(GetResponseSignup_Config::GETRESPONSE_360_DOMAIN, '');
    }

    /**
     * @return GetResponseSignup_Main
     */
    public static function getHandlerInstance() {
        return new GetResponseSignup_Main();
    }

    private function connect() {
        $this->apiKey = Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_API_KEY);
        $this->client = $this->getClientObject();
    }

    private function checkRequiredSettings() {
        if (!strlen(Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_API_KEY))) {
            $this->log('Api Key not defined. Please edit api key in GetResonseSignup plugin configuration!', Gpf_Log::ERROR);
            return false;
        }
        if (!strlen(Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_CAMPAIGN_NAME))) {
            $this->log('Campaign name not defined. Please edit Campaign name in GetResonseSignup plugin configuration', Gpf_Log::ERROR);
            return false;
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::OLD_360_API && !strlen(Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360_API_URL))) {
            $this->log('API URL not defined. Please edit GetResponse360 API URL in GetResonseSignup plugin configuration or set to common GetResponse', Gpf_Log::ERROR);
            return false;
        }
        if ($this->is360V3Api() && !strlen(Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360_DOMAIN))) {
            $this->log('API domain not defined. Please edit GetResponse360 API domain in GetResonseSignup plugin configuration or set to common GetResponse', Gpf_Log::ERROR);
            return false;
        }
        return true;
    }

    private function is360V3Api() {
        return Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::COM_V3_360_API || Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::PL_V3_360_API;
    }

    protected function getClientObject() {
        if (Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::OLD_360_API) {
            return new GetResponseSignup_JsonRPCClient($this->getGetResponseApiUrl(), $this->apiKey);
        }
        return new GetResponseSignup_V3Client($this->getGetResponseApiUrl(), $this->apiKey, $this->is360V3Api(), Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360_DOMAIN));
    }

    private function getGetResponseApiUrl() {
        if (Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::OLD_360_API) {
            return Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360_API_URL);
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::PL_V3_360_API) {
            return GetResponseSignup_Config::PL_V3_360_API_URL;
        }
        if (Gpf_Settings::get(GetResponseSignup_Config::GETRESPONSE_360) == GetResponseSignup_Config::COM_V3_360_API) {
            return GetResponseSignup_Config::COM_V3_360_API_URL;
        }
        return GetResponseSignup_Config::API_URL;
    }

    public function userStatusChanged(Gpf_Plugins_ValueContext $context) {
        $data = $context->get();
        $user = $data[0];
        $newStatus = $data[1];
        $this->log('userStatusChanged started, status:' . $newStatus, Gpf_Log::INFO);
        $oldEmail = $this->loadContactEmail($user);
        if ($newStatus == Pap_Common_Constants::STATUS_APPROVED && !$this->isSubscribed) {
            $this->signupToGetResponse($user);
            return;
        }
        if ($this->isSubscribed) {
            $this->deleteContact($user, $oldEmail);
        }
    }

    public function changeEmail(Pap_Common_User $user) {
        $this->log('changeEmail started', Gpf_Log::INFO);
        if (!$this->checkRequiredSettings()) {
            return;
        }
        $oldEmail = $this->loadContactEmail($user);
        if(!$this->isSubscribed) {
            if($user->getStatus() == Pap_Common_Constants::STATUS_APPROVED) {
                $this->signupToGetResponse($user);
            }
            return;
        }
        $this->log('Old email found: ' . $oldEmail, Gpf_Log::INFO);
        if($oldEmail == $user->getEmail() || $oldEmail == '') {
            return;
        }
        $this->connect();
        $this->client->changeEmail($user, $oldEmail);
    }

    public function changeEmailAuthUser(Gpf_Db_AuthUser $authUser) {
        try {
            $affiliateUser = Pap_Affiliates_User::loadFromUsername($authUser->getUsername());
        } catch (Gpf_Exception $e) {
            return;
        }
        $this->changeEmail($affiliateUser);
    }

    public function userDeleted(Pap_Common_User $user) {
        $this->log('userDeleted start.', Gpf_Log::INFO);
        if (!$this->checkRequiredSettings()) {
            return;
        }
        $this->loadContactEmail($user);
        $this->deleteContact($user, $user->getEmail());
    }

    private function deleteContact(Pap_common_user $user, $oldEmail) {
        if(!$this->isSubscribed) {
            return;
        }
        $this->log('deleteContact: '.$oldEmail);
        $this->connect();
        $this->client->deleteContact($oldEmail);

        $userAttr = new Gpf_Db_UserAttribute();
        $userAttr->setName(self::GET_RESPONSE_CONTACT_EMAIL);
        $userAttr->setAccountUserId($user->getAccountUserId());
        $userAttr->loadFromData();
        $userAttr->delete();
    }

    public function signupToGetResponse(Pap_Common_User $user) {
        $this->log('Signup started', Gpf_Log::INFO);
        if (!$this->checkRequiredSettings()) {
            return;
        }
        $this->connect();
        $this->loadContactEmail($user);
        if($this->isSubscribed) {
            $this->log('user has been already saved.', Gpf_Log::INFO);
            return;
        }
        if($user->getAccountUserId() == '') {
            $this->log('user has not been saved yet, returning', Gpf_Log::INFO);
            return;
        }
        $this->signup($user);
        $this->log('Signup end', Gpf_Log::INFO);
    }

    private function signup(Pap_Common_User $user) {
        if (!$this->checkRequiredSettings()) {
            return;
        }
        $this->getClientObject()->signup($user);
        $this->log('Affiliate added: '.$user->getEmail());
    }

    private function loadContactEmail(Pap_Common_User $user) {
        $this->log('loadContactEmail from userAttributes DB table', Gpf_Log::INFO);
        $userAttr = $this->getUserAttributeObject();
        $userAttr->setAccountUserId($user->getAccountUserId());
        $userAttr->setName(self::GET_RESPONSE_CONTACT_EMAIL);
        $this->isSubscribed = true;
        try {
            $userAttr->loadFromData();
        } catch (Gpf_Exception $e) {
            $this->log('contact not found in DB', Gpf_Log::INFO);
            $this->isSubscribed = false;
            return;
        }

        return $userAttr->getValue();
    }

    private function getUserAttributeObject() {
        return new Gpf_Db_UserAttribute();
    }

    /**
     * @return Gpf_Log_Logger
     */
    private function getLogger() {
        if ($this->logger === null) {
            $this->logger = Pap_Logger::create(Pap_Common_Constants::TYPE_SIGNUP);
        }
        return $this->logger;
    }

    private function log($message, $logLevel = Gpf_Log::DEBUG) {
        $this->getLogger()->log($this->getPluginName().': '.$message, $logLevel);
    }
}
