<?php
/**
 *   @copyright Copyright (c) 2014 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class BigCommerceAPI_Tracker extends Pap_Tracking_CallbackTracker {

    protected $xml = '';
    protected $couponXml = '';
    protected $xmlerror = '';

    private $productsXml = array();
    private $orderXml;

    private $isLoadedOrderXml = false;

    /**
     * @return BigCommerceAPI_Tracker
     */
    public static function getInstance() {
        $tracker = new BigCommerceAPI_Tracker();
        $tracker->setTrackerName("BigCommerceAPI");
        return $tracker;
    }

    public function checkStatus() {
        if ($this->getPaymentStatus() == 'OK') {
            return true;
        }
        $this->debug(' Invalid status received: ' . $this->getPaymentStatus());
        return false;
    }

    protected function computeTotalCost(SimpleXMLElement $product) {
        $totalCost = ((float) $product->base_price * (float) $product->quantity);
        $totalDiscount = 0;
        if (isset($product->applied_discounts->discount)) {
            foreach ($product->applied_discounts->discount as $discount) {
                if (isset($discount->amount)) {
                    $totalDiscount += (float) $discount->amount;
                }
            }
        }
        return ($totalCost - $totalDiscount);
    }

    /**
     *  @return Pap_Tracking_Request
     */
    protected function getRequestObject() {
        return Pap_Contexts_Action::getContextInstance()->getRequestObject();
    }

    protected function sendXML($url, $authentication, $method = 'products') {
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
        curl_setopt($curl, CURLOPT_USERPWD, $authentication);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 5);
        curl_setopt($curl, CURLOPT_TIMEOUT, 10);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array('Accept: application/xml','Content-Type: application/xml'));//DEPRECATED
        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_URL, $url);

        $response = curl_exec($curl);
        $error = curl_error($curl);
        curl_close($curl);

        if (!$response) {
            $this->xmlerror = $error;
            return false;
        }

        $response = str_replace('<![CDATA[', '', $response);
        $response = str_replace(']]>', '', $response);
        switch ($method) {
            case 'coupons':
                $this->couponXml = $response;
                break;
            default:
                $this->xml = $response;
        }
        return true;
    }

    protected function getApiPath() {
        return Gpf_Settings::get(BigCommerceAPI_Config::API_PATH);
    }

    protected function getApiAuthentication() {
        return Gpf_Settings::get(BigCommerceAPI_Config::API_USERNAME) . ':' . Gpf_Settings::get(BigCommerceAPI_Config::API_TOKEN);
    }

    protected function getCouponTrackingSetting() {
        return Gpf_Settings::get(BigCommerceAPI_Config::COUPON_TRACKING);
    }

    protected function getPerProductTrackingSetting() {
        return Gpf_Settings::get(BigCommerceAPI_Config::PER_PRODUCT);
    }

    public function readRequestVariables() {
        $request = $this->getRequestObject();
        $orderId = $request->getRequestParameter('orderId');
        $cookie = $request->getRequestParameter('visitorId');
        $email = $request->getRequestParameter('email');

        $this->debug(" received order with ID $orderId, the cookie value is $cookie and customer's email is $email");

        $apiPath = $this->getApiPath();
        if ($apiPath == '') {
            $this->debug(' Please configure your plugin first! The API Path is missing!');
            $this->setPaymentStatus('FAILED');
            return false;
        }

        if (empty($orderId)) {
            $this->setPaymentStatus('EMPTY');
            return false;
        }
        $this->setPaymentStatus('OK');

        $this->setCookie($cookie);
        $this->setTransactionID($orderId);

        $url = $apiPath . 'orders/' . $orderId . '/products?limit=250';
        $authentication = $this->getApiAuthentication();

        $this->debug(" Sending an XML request to $url");
        if ($this->sendXML($url, $authentication)) {
            $this->debug(' XML response received: ' . $this->xml);
            $this->xml = preg_replace_callback("/&[^; ]{0,6}.?/", function($matches) {return ((substr($matches[0],-1) == ';') ? $matches[0] : '&amp;'.substr($matches[0],1));}, $this->xml);
            try {
                @$response = new SimpleXMLElement($this->xml);
            } catch (Exception $e) {
                $this->error(' Decoding xml error: '.$e->getMessage().', loaded xml: ' . $this->xml);
                $this->setPaymentStatus('FAILED');
                return false;
            }
            $wasCouponUsed = false;
            foreach ($response as $product) {
                $this->productsXml[] = $product;
                if (!empty($product->applied_discounts)) {
                    $wasCouponUsed = true;
                }
            }
            if ($this->getCouponTrackingSetting() == Gpf::YES && $wasCouponUsed) {
                $url = $apiPath . 'orders/' . $orderId . '/coupons';
                $this->sendXML($url, $authentication, 'coupons');
                if (!($XMLObj = $this->getXMLObject('coupons'))) {
                    $this->error('unable to load coupons xml object');
                } else {
                    $this->setCoupon((string) $XMLObj->coupon->code);
                }
            }

            if (!$this->areDataFieldsEmpty()) {
                $this->fillDataFields($request);
            }

            if (!$this->isLoadedOrderXml && ($this->getPerProductTrackingSetting() != Gpf::YES || Gpf_Settings::get(Pap_Settings::MULTIPLE_CURRENCIES) == Gpf::YES)) {
                $this->loadOrderDetails($authentication, $orderId);
            }
        } else {
            $this->error(' An error occurred when communicating with the remote server: ' . $this->xmlerror);
            $this->setPaymentStatus('FAILED');
            return false;
        }
        if (isset($response->error)) {
            $this->error(' An error occurred when communicating with the remote server: ' . $response->error->message);
            $this->setPaymentStatus('FAILED');
        }
    }

    private function fillDataFields(Pap_Tracking_Request $request) {
        for ($i = 1; $i <= 5; $i++) {
            $dataFieldName = Gpf_Settings::get(BigCommerceAPI_Config::DATA.$i);
            $dataValue = $this->getDataFieldValue($dataFieldName, $request);
            if ($dataValue != '') {
                $this->setData($i, $dataValue);
            }
        }
    }

    private function getDataFieldValue($dataFieldName, Pap_Tracking_Request $request) {
        if ($dataFieldName == '' || $dataFieldName == '0') {
            return '';
        }
        if ($dataFieldName == 'customer_email') {
            $emailRequestParameter = $request->getRequestParameter('email');
            if ($emailRequestParameter != '') {
                return $emailRequestParameter;
            }
        }
        if ($this->isProductDataFieldName($dataFieldName)) {
            return '';
        }

        if (!$this->isLoadedOrderXml) {
            $this->loadOrderDetails($this->getApiAuthentication(), $this->getTransactionID());
            $this->isLoadedOrderXml = true;
        }
        return $this->getOrderDataValue($dataFieldName);
    }

    private function areDataFieldsEmpty() {
        return ((Gpf_Settings::get(BigCommerceAPI_Config::DATA1) == '' || Gpf_Settings::get(BigCommerceAPI_Config::DATA1) == '0') &&
                (Gpf_Settings::get(BigCommerceAPI_Config::DATA2) == '' || Gpf_Settings::get(BigCommerceAPI_Config::DATA2) == '0') &&
                (Gpf_Settings::get(BigCommerceAPI_Config::DATA3) == '' || Gpf_Settings::get(BigCommerceAPI_Config::DATA3) == '0') &&
                (Gpf_Settings::get(BigCommerceAPI_Config::DATA4) == '' || Gpf_Settings::get(BigCommerceAPI_Config::DATA4) == '0') &&
                (Gpf_Settings::get(BigCommerceAPI_Config::DATA5) == '' || Gpf_Settings::get(BigCommerceAPI_Config::DATA5) == '0'));
    }

    private function isProductDataFieldName($dataFieldName) {
        $productFieldNames = array('product_id', 'name','sku','base_price','quantity');
        return in_array($dataFieldName, $productFieldNames);
    }

    private function getOrderDataValue($fieldName) {
        if ($this->isCustomFormField($fieldName)) {
            return $this->getCustomFormField($fieldName);
        }
        switch ($fieldName) {
            case 'customer_name':
                if ($this->orderXml != null && isset($this->orderXml->billing_address)) {
                    return (string)$this->orderXml->billing_address->first_name.' '.(string)$this->orderXml->billing_address->last_name;
                }
                $this->error('Error loading customer name, loaded response: ' .var_export($this->orderXml, true));
                return '';
            case 'customer_email':
                if ($this->orderXml != null && isset($this->orderXml->billing_address)) {
                    return (string)$this->orderXml->billing_address->email;
                }
                $this->error('Error loading customer email, loaded response: ' .var_export($this->orderXml, true));
                return '';
            case 'customer_country':
                if ($this->orderXml != null && isset($this->orderXml->billing_address)) {
                    return (string)$this->orderXml->billing_address->country;
                }
                $this->error('Error loading customer country, loaded response: ' .var_export($this->orderXml, true));
                return '';
        }
        return '';
    }

    private function getProductDataValue($fieldName, $product) {
        if ($fieldName == '') {
            return '';
        }
        switch ($fieldName) {
            case 'product_id':
                return (string) $product->product_id;
            case 'name':
                return (string) $product->name;
            case 'sku':
                return (string) $product->sku;
            case 'base_price':
                return (float) $product->base_price;
            case 'quantity':
                return (float) $product->quantity;
        }
        return '';
    }

    private function getCustomFormField($fieldName) {
        if ($this->orderXml == null || !isset($this->orderXml->billing_address) || !isset($this->orderXml->billing_address->form_fields)) {
            return '';
        }
        foreach ($this->orderXml->billing_address->form_fields->form_fields as $formField) {
            if ((string)$formField->name == $fieldName) {
                return (string)$formField->value;
            }
        }
        return '';
    }

    private function isCustomFormField($fieldName) {
        if ($this->orderXml == null || !isset($this->orderXml->billing_address) || !isset($this->orderXml->billing_address->form_fields)) {
            return false;
        }
        foreach ($this->orderXml->billing_address->form_fields->form_fields as $formField) {
            if ((string)$formField->name == $fieldName) {
                return true;
            }
        }
        return false;
    }

    private function loadOrderDetails($authentication, $orderId) {
        $url = $this->getApiPath() . 'orders/'.$orderId;
        $this->debug(" Sending an XML request to $url");
        if ($this->sendXML($url, $authentication)) {
            $XMLObj = $this->getXMLObject();
            if (isset($XMLObj[0])) {
                $this->orderXml = $XMLObj[0];
            } else {
                $this->error('Unable to load detail for orderid: ' . $orderId . ', loaded response: ' . var_export($XMLObj, true));
            }
        } else {
            $this->error('An error occurred when loading order detail for orderid: ' . $orderId . ', error: ' .$this->xmlerror);
        }
    }

    public function getOrderID() {
        return $this->getTransactionID();
    }

    protected function prepareSales(Pap_Tracking_ActionTracker $saleTracker) {
        if ($this->orderXml !== null) {
            $this->setCurrency((string) $this->orderXml->currency_code);
        }
        if ($this->getPerProductTrackingSetting() == Gpf::YES) {
            $this->prepareSeparateCartItems($saleTracker);
        } else {
            $total = 0;
            $productIds = '';
            if (count($this->productsXml) == 0) {
                return;
            }
            $totalExcludeTax = null;
            if ($this->orderXml !== null) {
                $totalExcludeTax = (float) $this->orderXml->subtotal_ex_tax;
                $couponDiscount = (float) $this->orderXml->coupon_discount;
                $discountAmount = (float) $this->orderXml->discount_amount;
                $totalExcludeTax -= $couponDiscount;
                $totalExcludeTax -= $discountAmount;
            }

            $data1ProductValue = '';
            $data2ProductValue = '';
            $data3ProductValue = '';
            $data4ProductValue = '';
            $data5ProductValue = '';

            foreach ($this->productsXml as $product) {
                if ($totalExcludeTax === null) {
                    $total += $this->computeTotalCost($product);
                }

                $productId = (string) $product->sku;
                if ($productId != '') {
                    $productIds .= $productId . ';';
                }

                if ($this->getData1() == '') {
                    $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA1), $product);
                    if ($dataValue != '') {
                        $data1ProductValue .= $dataValue . ';';
                    }
                }
                if ($this->getData2() == '') {
                    $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA2), $product);
                    if ($dataValue != '') {
                        $data2ProductValue .= $dataValue . ';';
                    }
                }
                if ($this->getData3() == '') {
                    $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA3), $product);
                    if ($dataValue != '') {
                        $data3ProductValue .= $dataValue . ';';
                    }
                }
                if ($this->getData4() == '') {
                    $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA4), $product);
                    if ($dataValue != '') {
                        $data4ProductValue .= $dataValue . ';';
                    }
                }
                if ($this->getData5() == '') {
                    $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA5), $product);
                    if ($dataValue != '') {
                        $data5ProductValue .= $dataValue . ';';
                    }
                }
            }
            $sale = $saleTracker->createSale();
            $sale->setTotalCost(($totalExcludeTax === null ? $total : $totalExcludeTax));
            $this->debug('Computed total cost: ' . $sale->getTotalCost());
            $sale->setOrderID($this->getOrderID());
            $productIds = rtrim($productIds, ';');
            if (strlen($productIds) > 255) {
                $productIds = substr($productIds, 0, 252).'...';
            }
            $sale->setProductID($productIds);

            if ($data1ProductValue != '') {
                if (strlen($data1ProductValue) > 255) {
                    $data1ProductValue = substr($data1ProductValue, 0, 252).'...';
                }
                $sale->setData1($data1ProductValue);
            } else {
                $sale->setData1($this->getData1());
            }
            if ($data2ProductValue != '') {
                if (strlen($data2ProductValue) > 255) {
                    $data2ProductValue = substr($data2ProductValue, 0, 252).'...';
                }
                $sale->setData2($data2ProductValue);
            } else {
                $sale->setData2($this->getData2());
            }
            if ($data3ProductValue != '') {
                if (strlen($data3ProductValue) > 255) {
                    $data3ProductValue = substr($data3ProductValue, 0, 252).'...';
                }
                $sale->setData3($data3ProductValue);
            } else {
                $sale->setData3($this->getData3());
            }
            if ($data4ProductValue != '') {
                if (strlen($data4ProductValue) > 255) {
                    $data4ProductValue = substr($data4ProductValue, 0, 252).'...';
                }
                $sale->setData4($data4ProductValue);
            } else {
                $sale->setData4($this->getData4());
            }
            if ($data5ProductValue != '') {
                if (strlen($data5ProductValue) > 255) {
                    $data5ProductValue = substr($data5ProductValue, 0, 252).'...';
                }
                $sale->setData5($data5ProductValue);
            } else {
                $sale->setData5($this->getData5());
            }

            $sale->setCoupon($this->getCouponCode());
            $sale->setCurrency($this->getCurrency());

            $this->setVisitorAndAccount($saleTracker, $this->getAffiliateID(), $this->getCampaignID(), $this->getCookie());
        }
    }

    protected function prepareSeparateCartItems(Pap_Tracking_ActionTracker $saleTracker) {
        $i = 1;
        foreach ($this->productsXml as $product) {
            $sale = $saleTracker->createSale();
            $sale->setTotalCost($this->computeTotalCost($product));
            $sale->setOrderID($this->getOrderID() . '(' . $i . ')');
            $sale->setProductID((string) $product->sku);
            if ($this->getData1() != '') {
                $sale->setData1($this->getData1());
            } else {
                $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA1), $product);
                if ($dataValue != '') {
                    $sale->setData1($dataValue);
                }
            }
            if ($this->getData2() != '') {
                $sale->setData2($this->getData2());
            } else {
                $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA2), $product);
                if ($dataValue != '') {
                    $sale->setData2($dataValue);
                }
            }
            if ($this->getData3() != '') {
                $sale->setData3($this->getData3());
            } else {
                $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA3), $product);
                if ($dataValue != '') {
                    $sale->setData3($dataValue);
                }
            }
            if ($this->getData4() != '') {
                $sale->setData4($this->getData4());
            } else {
                $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA4), $product);
                if ($dataValue != '') {
                    $sale->setData4($dataValue);
                }
            }
            if ($this->getData5() != '') {
                $sale->setData5($this->getData5());
            } else {
                $dataValue = $this->getProductDataValue(Gpf_Settings::get(BigCommerceAPI_Config::DATA5), $product);
                if ($dataValue != '') {
                    $sale->setData5($dataValue);
                }
            }
            $sale->setCoupon($this->getCouponCode());
            $sale->setCurrency($this->getCurrency());

            $this->setVisitorAndAccount($saleTracker, $this->getAffiliateID(), $this->getCampaignID(), $this->getCookie());
            $i++;
        }
    }

    protected function getXMLObject($method = 'products') {
        try {
            switch ($method) {
                case 'coupons':
                    @$XMLObj = new SimpleXMLElement($this->couponXml);
                    break;
                default:
                    @$XMLObj = new SimpleXMLElement($this->xml);
            }
        } catch (Exception $e) {
            return false;
        }
        return $XMLObj;
    }
}
