<?php
/**
 *   @copyright Copyright (c) 2015 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.qualityunit.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class Stripe_Tracker extends Pap_Tracking_CallbackTracker {
    const TRANSACTION_TYPE_REFUND = 'charge.refunded';
    const TRANSACTION_TYPE_APPROVED = 'charge.succeeded';
    const TRANSACTION_TYPE_INVOICE_APPROVED = 'invoice.payment_succeeded';
    const TRANSACTION_TYPE_CHECKOUT_SESSION_COMPLETED = 'checkout.session.completed';

    /**
     * @return Stripe_Tracker
     */
    public static function getInstance() {
        $tracker = new Stripe_Tracker();
        $tracker->setTrackerName('Stripe');
        return $tracker;
    }

    private function getTransactionIdFromOrderId($orderId){
        $transaction = new Pap_Common_Transaction();
        if ($output = $this->findFirstRecordWithData($transaction, Pap_Db_Table_Transactions::ORDER_ID, $orderId)) {
            $this->debug('Parent transaction for refund found by orderId.');
            return $output->getId();
        }
        // try subscription ID now as well
        $subID = $this->getSubscriptionIDFromStripe($orderId);
        if ($subID) {
            if ($output = $this->findFirstRecordWithData($transaction, Pap_Db_Table_Transactions::ORDER_ID, $subID)) {
                $this->debug('Parent transaction for refund found by subscription ID.');
                return $output->getId();
            }
        }

        throw new Gpf_Exception('Parent transaction for order id: ' . $orderId . ' not found.');
    }

    protected function refundChargeback() {
        $transaction = new Pap_Common_Transaction();
        $transaction->processRefundChargeback($this->getTransactionIdFromOrderId($this->getOrderID()), Pap_Db_Transaction::TYPE_REFUND, '', $this->getOrderID(), 0, true);
    }

    public function checkStatus() {
        $this->sendBackVerification();

        if ($this->getType() == self::TRANSACTION_TYPE_REFUND) {
            $this->debug('Transaction '.$this->getOrderID().' will be refunded');
            try {
                $this->refundChargeback();
                $this->debug('Refund completed, ending process.');
            } catch (Gpf_Exception $e) {
                $this->debug('Error occurred during transaction refund:' . $e->getMessage());
            }
            return false;
        }

        if (Gpf_Settings::get(Stripe_Config::TRACK_CHARGE_EVENT) == Gpf::YES) {
            if (($this->getType() == self::TRANSACTION_TYPE_APPROVED)) {
                return true;
            }
        }

        if (Gpf_Settings::get(Stripe_Config::TRACK_SESSION_EVENT) == Gpf::YES) {
            if (($this->getType() == self::TRANSACTION_TYPE_CHECKOUT_SESSION_COMPLETED)) {
                return true;
            }
        }

        if (($this->getType() == self::TRANSACTION_TYPE_INVOICE_APPROVED)) {
            return true;
        }

        $this->debug("Ignoring type '".$this->getType()."'");
        return false;
    }

    protected function sendBackVerification() {
        return "200";
    }

    /**
     *  @return Pap_Tracking_Request
     */
    protected function getRequestObject() {
        $response = @file_get_contents('php://input');
        $response_object = json_decode($response);

        $this->debug('callback data: '.print_r($response_object,true));
        return $response_object;
    }

    private function isAccountIdValid($accountId) {
        if (Gpf_Db_Account::DEFAULT_ACCOUNT_ID == $accountId) {
            return true;
        }
        if ($accountId == '') {
            return false;
        }
        $select = new Gpf_SqlBuilder_SelectBuilder();
        $select->select->addDontQuote('*');
        $select->from->add(Gpf_Db_Table_Accounts::getName());
        $select->where->add(Gpf_Db_Table_Accounts::ID, '=', $accountId);
        $select->limit->set(0, 1);
        try {
            $select->getOneRow();
            return true;
        } catch (Gpf_Exception $e) {
        }
        return false;
    }

    private function getIntegrationAccountId() {
        $request = array_change_key_case($_GET, CASE_LOWER);
        $paramName = 'accountid';
        if (isset($request[$paramName]) && $this->isAccountIdValid($request[$paramName])) {
            return $request[$paramName];
        }
        return Gpf_Db_Account::DEFAULT_ACCOUNT_ID;
    }

    protected function loadValuesForExtraData($request) {
        if ($this->getType() == self::TRANSACTION_TYPE_INVOICE_APPROVED) {
            // get charge object for easier work
            $key = $this->getAPIkey();

            if ($request->charge == '' || $request->charge == null) {
                $this->error('Error loading charge: No charge object/ID found');
                return false;
            }

            try {
                $request = Stripe\Charge::retrieve($request->charge, $key);
                $this->debug('Charge loaded: '.var_export($request, true));
            } catch (Stripe\InvalidRequestError $e) {
                $this->error('Error loading charge (InvalidRequest): '.$e->json_body['error']['message']);
                return false;
            } catch (Stripe\AuthenticationError $e) {
                $this->error('Error loading charge (AuthenticationError): '.$e->json_body['error']['message']);
                return false;
            } catch (Stripe\Error $e) {
                $this->error('Error loading charge: '.$e->json_body['error']['message']);
                return false;
            } catch (Exception $e) {
                $this->error('Error loading customer: '.$e->getMessage());
                return false;
            }
        }

        for ($i = 1; $i <= 5; $i++) {
            if ($i == 2) continue; // data2 reserverd for transaction ID
            $stripeChargeValue = $this->getChargeValueFromIndex($request, Gpf_Settings::get(constant('Stripe_Config::DATA'.$i)));
            $this->setData($i, $stripeChargeValue);
        }

        if (Gpf_Settings::get(Stripe_Config::CREATE_AFFILIATE) == Gpf::YES) {
            $this->loadCustomerValues($request);
        }
    }

    private function getChargeValueFromIndex($request, $customerValueIndex) {
        if (!isset($request->billing_details)) {
            return '';
        }
        switch ($customerValueIndex) {
            case 'email': $email = $this->getEmail();
                if ($email == ''){
                    $email = $request->billing_details->email;
                }
                return $email;
            case 'id': return $request->customer;
            case 'customer_phone': return $request->billing_details->phone;
            case 'customer_name': return $request->billing_details->name;
            case 'address_street': return $request->billing_details->address->line1;
            case 'address_city': return $request->billing_details->address->city;
            case 'address_state': return $request->billing_details->address->state;
            case 'address_country': return $request->billing_details->address->country;
            case 'address_zip': return $request->billing_details->address->postal_code;
            default: return '';
        }
    }

    private function loadCustomerValues($request) {
        $email = $this->getEmail();
        if ($email == ''){
            $email = $request->billing_details->email;
        }
        $this->setUserEmail($email);
        if (strpos($request->billing_details->name, ' ')) {
            $this->setUserFirstName(substr($request->billing_details->name, 0, strpos($request->billing_details->name, ' ')));
            $this->setUserLastName(substr($request->billing_details->name, strpos($request->billing_details->name, ' ')+1));
        } else {
            $this->setUserFirstName(trim($request->billing_details->name));
        }
        $this->setUserAddress($request->billing_details->address->line1);
        $this->setUserCity($request->billing_details->address->city);
    }

    public function readRequestVariables() {
        $request = $this->getRequestObject();
        if (empty($request)) {
            return;
        }

        $ingetrationAccountId = $this->getIntegrationAccountId();
        $this->debug('tracking accountID: '.$ingetrationAccountId);
        Gpf_Session::getAuthUser()->setAccountId($ingetrationAccountId);

        $this->setType((string)$request->type);
        $this->debug('Webhook version: '.$request->api_version);

        if (($this->getType() != self::TRANSACTION_TYPE_INVOICE_APPROVED) &&
                ($this->getType() != self::TRANSACTION_TYPE_REFUND) &&
                ($this->getType() != self::TRANSACTION_TYPE_APPROVED) &&
                ($this->getType() != self::TRANSACTION_TYPE_CHECKOUT_SESSION_COMPLETED)) {
            return;
        }

        if (Gpf_Settings::get(Stripe_Config::TRACK_CHARGE_EVENT) == Gpf::YES) {
            if (($this->getType() == self::TRANSACTION_TYPE_INVOICE_APPROVED)) {
                return;
            }
        }

        require_once('lib/Stripe.php');
        $request = $request->data->object;

        $customer = $this->getCustomer($request->customer);
        if ($customer !== false && isset($customer->description)) {
            $cookieValue = $customer->description;
        } else {
            $cookieValue = '';
        }

        if ($this->getType() == self::TRANSACTION_TYPE_CHECKOUT_SESSION_COMPLETED) {
            // no billing details here, just item ordered
            $this->setCookie($request->client_reference_id);
            $this->setTransactionID((string)$request->id);
            $this->setEmail($this->getEmail());

            $subscriptionId = @$request->subscription;
            $this->setSubscriptionID($subscriptionId);
            $this->setData2($subscriptionId);
            if (isset($request->display_items[0])) {
                $this->setCurrency($request->display_items[0]->currency);
                $this->setTotalCost($this->adjustTotalCost((float)$request->display_items[0]->amount));
                if (isset($request->display_items[0]->plan)) {
                    $productIdSetting = Gpf_Settings::get(Stripe_Config::PRODUCT_ID);
                    $this->setProductID($request->display_items[0]->plan->$productIdSetting);
                }
            }
            return true;
        }

        // load values for extra data
        $this->loadValuesForExtraData($request);

        try {
            $customSeparator = Gpf_Settings::get(Stripe_Config::CUSTOM_SEPARATOR);
            if ($customSeparator != '') {
                $explodedCookieValue = explode($customSeparator, $cookieValue, 2);
                if (count($explodedCookieValue) == 2) {
                    $cookieValue = $explodedCookieValue[1];
                }
            }
        } catch (Gpf_Exception $e) {
        }

        $this->setCookie($cookieValue);
        $discount = @$request->discount;
        if (!empty($discount)) {
            $coupon = (string)$request->discount->coupon->id;
            if (!empty($coupon)) {
                if (strpos($coupon,'_') !== false) {
                    $this->setCoupon(substr($coupon,0,strpos($coupon,"_")));
                } else {
                    $this->setCoupon($coupon);
                }
            }
        }

        $this->setTransactionID((string)$request->id);

        $subscriptionId = @$request->subscription;

        $lines = @$request->lines;
        if (!empty($lines)) {
            if (!$this->isSubscriptionID($subscriptionId)) {
                $subscriptionId = $this->findSubscriptionID($lines);
            }
        }

        $this->setSubscriptionID($subscriptionId);
        $this->setData2($subscriptionId);

        $productId = '';
        if (!empty($lines)) {
            // check if exists, if not, create a new transaction with subscription ID instead of invoice ID
            if (!$this->checkIfTransactionExists($subscriptionId)) {
                $this->setTransactionID($subscriptionId);
            }
            $productIdSetting = Gpf_Settings::get(Stripe_Config::PRODUCT_ID);
            if ($lines->data[0]->plan != '' && isset($lines->data[0]->plan->$productIdSetting)) {
                $productId = (string)$lines->data[0]->plan->$productIdSetting;
            }
        }

        if (isset($request->currency)) {
            $currency = (string)$request->currency;
            $this->setCurrency(strtoupper($currency));
        }

        if ($this->getType() == self::TRANSACTION_TYPE_REFUND) {
            $this->setTotalCost($this->adjustTotalCost((float)$request->amount_refunded));
            if (Gpf_Settings::get(Stripe_Config::TRACK_CHARGE_EVENT) === Gpf::NO) {
                $this->setTransactionID((string)$request->invoice);
            }
        }
        if ($this->getType() == self::TRANSACTION_TYPE_APPROVED) {
            $this->setTotalCost($this->adjustTotalCost((float)$request->amount));
            $this->setProductID((string)$request->description);
        }
        if ($this->getType() == self::TRANSACTION_TYPE_INVOICE_APPROVED) {
            $this->setTotalCost($this->adjustTotalCost((float)$request->amount_due));
            if ($productId) {
                $this->setProductID($productId);
            }
        }
    }

    private function isSubscriptionID($value) {
        if (gettype($value) == 'object' ||
                gettype($value) == 'array' ||
                gettype($value) == 'resource') {
            return false;
        }
        if (empty($value) || (substr($value, 0, 4) != 'sub_')) {
            return false;
        }
        return true;
    }

    private function findSubscriptionID($lines) {
        $index = Gpf_Settings::get(Stripe_Config::SUBSCRIPTION_INDEX);
        $subscriptionId = (string)$lines->data[0]->{$index};
        if ($this->isSubscriptionID($subscriptionId)) {
            return $subscriptionId;
        }
        // let's identify the index of field where subscription ID is held in
        foreach ($lines->data[0] as $key => $value) {
            if ($this->isSubscriptionID($value)) {
                // save the value to plugin and return subscription ID
                Gpf_Settings::set(Stripe_Config::SUBSCRIPTION_INDEX, $key);
                return $value;
            }
        }

    }

    protected function isAffiliateRegisterAllowed() {
        return (Gpf_Settings::get(Stripe_Config::CREATE_AFFILIATE) == Gpf::YES);
    }

    private function adjustTotalCost($total) {
        // zero-decimal currencies do not need division by 100
        // official source: https://support.stripe.com/questions/which-zero-decimal-currencies-does-stripe-support
        $zeroDecimals = array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','VND','VUV','XAF','XOF','XPF');
        if (!in_array($this->getCurrency(),$zeroDecimals)) {
            return (float)$total/100;
        }
        return $total;
    }

    /**
     * @param String $orderId
     * @return String|boolean
     */
    protected function checkIfTransactionExists($orderId) {
        $transaction = new Pap_Common_Transaction();
        try {
            $result = $transaction->getFirstRecordWith(Pap_Db_Table_Transactions::ORDER_ID, $orderId, array(Pap_Common_Constants::STATUS_APPROVED, Pap_Common_Constants::STATUS_PENDING));
            return $result->getId();
        } catch (Gpf_DbEngine_NoRowException $e) {
            return false;
        }
    }

    protected function getCustomer($cust_id) {
        $this->debug('Loading customer for customer ID: '.$cust_id);
        if ($cust_id == '') {
            $this->error('Loading customer error, empty customer id');
            return false;
        }
        $key = $this->getAPIkey();

        try {
            $customer = Stripe\Customer::retrieve($cust_id, $key);

            $this->debug('Customer found: '.var_export($customer,true));
            if (!isset($customer->email) && !isset($customer->description)) {
                $this->error('Error! Both email and description fields are empty. Received customer object: '.var_export($customer,true));
                return false;
            }

            if (isset($customer->email)) {
                $this->setEmail($customer->email);
            }

            return $customer;
        } catch (Stripe\InvalidRequestError $e) {
            $this->error('Error loading customer (InvalidRequest): '.$e->json_body['error']['message']);
            return false;
        } catch (Stripe\AuthenticationError $e) {
            $this->error('Error loading customer (AuthenticationError): '.$e->json_body['error']['message']);
            return false;
        } catch (Stripe\Error $e) {
            $this->error('Error loading customer: '.$e->json_body['error']['message']);
            return false;
        } catch (Exception $e) {
            $this->error('Error loading customer: '.$e->getMessage());
            return false;
        }
    }

    protected function getSubscriptionIDFromStripe($invoiceId) {
        if (strpos($invoiceId, 'in_') != 0) { // this is not an invoice ID
            return false;
        }
        $this->debug(' Loading invoice '.$invoiceId);
        $key = $this->getAPIkey();
        require_once('lib/Stripe.php');

        try {
            $invoice = Stripe\Invoice::retrieve($invoiceId, $key);

            if (!isset($invoice->subscription)) {
                return false;
            }
            $this->debug(' Invoice found, subscription ID is '.$invoice->subscription);

            return $invoice->subscription;
        } catch (Stripe\InvalidRequestError $e) {
            $this->error('Error loading invoice (InvalidRequest): '.$e->json_body['error']['message']);
            return false;
        } catch (Stripe\AuthenticationError $e) {
            $this->error('Error loading invoice (AuthenticationError): '.$e->json_body['error']['message']);
            return false;
        } catch (Stripe\Error $e) {
            $this->error('Error loading invoice: '.$e->json_body['error']['message']);
            return false;
        } catch (Exception $e) {
            $this->error('Error loading customer: '.$e->getMessage());
            return false;
        }
    }

    private function getAPIkey() {
        $settingId = Stripe_Config::API_KEY;
        if (Gpf_Settings::get(Stripe_Config::ENVIRONMENT) == Stripe_Config::ENVIRONMENT_TEST) {
            $settingId = Stripe_Config::TEST_API_KEY;
        }

        return array('api_key' => Gpf_Settings::get($settingId));
    }

    public function getOrderID() {
        return $this->getTransactionID();
    }

    protected function getRecurringTotalCost() {
        return $this->getTotalCost();
    }

    public function isRecurring() {
        return $this->existRecurringCommission($this->getData2());
    }

    private function existRecurringCommission($orderId) {
        if (empty($orderId)) {
            return false;
        }

        return Pap_Features_RecurringCommissions_Main::isExistRecurringRule($orderId);
    }
}
