<?php
/**
 *   @copyright Copyright (c) 2018 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class Chargify_Tracker extends Pap_Tracking_CallbackTracker {
    const CHARGIFY_REFUND = 'refund_success';
    const CHARGIFY_PAYMENT = 'payment_success';

    private function getTransactionIdFromOrderId($orderId) {
        $transaction = new Pap_Common_Transaction();
        if ($output = $this->findFirstRecordWithData($transaction, Pap_Db_Table_Transactions::DATA2, $orderId)) {
            $this->debug('Parent transaction for refund found');
            return $output->getId();
        }

        throw new Gpf_Exception('Parent transaction for order id: ' . $orderId . ' not found.');
    }

    /**
     * @return Chargify_Tracker
     */
    public static function getInstance() {
        $tracker = new Chargify_Tracker();
        $tracker->setTrackerName('Chargify');
        return $tracker;
    }

    protected function refundChargeback() {
        $transaction = new Pap_Common_Transaction();
        $transaction->processRefundChargeback($this->getTransactionIdFromOrderId($this->getData2()), Pap_Db_Transaction::TYPE_REFUND, '', $this->getOrderID(), 0, true);
    }

    public function checkStatus() {
        if ($this->getType() == self::CHARGIFY_REFUND) {
            if (Gpf_Settings::get(Chargify_Config::TRACK_REFUNDS) != Gpf::YES) {
                $this->debug('Status "Refunded" received, but refund handling is not enabled, ending.');
                return false;
            }
            $this->debug('Transaction ' . $this->getOrderID() . ' will be marked as a refund');
            try {
                $this->refundChargeback();
                $this->debug('Refund completed, ending processing.');
            } catch (Gpf_Exception $e) {
                $this->debug('Error occurred during transaction refund:' . $e->getMessage());
            }
            return false;
        }

        if ($this->getType() == self::CHARGIFY_PAYMENT) {
            return true;
        } else {
            $this->debug('We do not track "' . $this->getType() . '" notifications.');
            return false;
        }
    }

    /**
     *  @return Pap_Tracking_Request
     */
    protected function getRequestObject() {
        return Pap_Contexts_Action::getContextInstance()->getRequestObject();
    }

    public function readRequestVariables() {
        $request = $this->getRequestObject();
        $this->debug('Request data received: ' . print_r($request, true));

        $this->setType($request->getRequestParameter('event'));
        $payload = $request->getRequestParameter('payload', false);

        if ($this->getType() == self::CHARGIFY_PAYMENT) {
            $amount = $payload['transaction']['amount_in_cents'] / 100;
            $transId = $payload['transaction']['id'];
            if ($payload['subscription']['signup_payment_id'] == $transId) {
                $this->setType('initial');
                return false;
            }
            $subId = $payload['subscription']['signup_payment_id'];
            $customerId = $payload['subscription']['customer']['id'];
            $prodId = $payload['transaction']['product_id'];
        } elseif ($this->getType() == self::CHARGIFY_REFUND) {
            $amount = $payload['amount_in_cents'] / 100;
            $subId = $payload['subscription_id'];
            $transId = $payload['refund_id'];
            $customerId = $payload['customer_id'];
            $prodId = '';
        } else {
            return; // not interested in other type of notification
        }

        $this->setTotalCost($amount);
        $this->setSubscriptionID($subId);
        $this->setData1($customerId);
        $this->setData2($subId);
        $this->readAdditionalVariables($request);
    }

    public function readAdditionalVariables(Pap_Tracking_Request $request) {
        if ($this->getData1() == '') {
            $this->setData1($request->getRequestParameter('data1'));
        }
        if ($this->getData2() == '') {
            $this->setData2($request->getRequestParameter('data2'));
        }
        if ($this->getData3() == '') {
            $this->setData3($request->getRequestParameter('data3'));
        }
        if ($this->getData4() == '') {
            $this->setData4($request->getRequestParameter('data4'));
        }
        if ($this->getData5() == '') {
            $this->setData5($request->getRequestParameter('data5'));
        }
        if ($this->getCouponCode() == '') {
            $this->setCoupon($request->getRequestParameter('coupon_code'));
        }
        if ($this->getChannelId() == '') {
            $this->setChannelId($request->getRequestParameter('channelId'));
        }
        if ($this->getAffiliateID() == '') {
            $this->setAffiliateID($request->getRequestParameter('AffiliateID'));
        }
        if ($this->getCampaignID() == '') {
            $this->setCampaignID($request->getRequestParameter('CampaignID'));
        }
    }

    public function isRecurring() {
        if ($this->getType() == self::CHARGIFY_REFUND) {
            return false;
        }
        return true;
    }

    protected function allowUseRecurringCommissionSettings() {
        return (Gpf_Settings::get(Chargify_Config::USE_RECURRING_COMMISSION_SETTINGS) == Gpf::YES);
    }

    protected function getRecurringTotalCost() {
        if (Gpf_Settings::get(Chargify_Config::RECURRING_TOTALCOST_FROM_NOTIFICATION) == Gpf::YES) {
            return $this->getTotalCost();
        }
        return null;
    }

    public function getOrderID() {
        return $this->getSubscriptionID();
    }
}
