<?php
/**
 *   @copyright Copyright (c) 2017 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class Chargebee_Tracker extends Pap_Tracking_CallbackTracker {

    private function getTransactionIdFromOrderId($orderId) {
        $transaction = new Pap_Common_Transaction();
        if ($output = $this->findFirstRecordWithData($transaction, Pap_Db_Table_Transactions::ORDER_ID, $orderId)) {
            $this->debug('Parent transaction for refund found by orderId.');
            return $output->getId();
        }
        
        throw new Gpf_Exception('Parent transaction for order id: ' . $orderId . ' not found.');
    }

    /**
     * @return Chargebee_Tracker
     */
    public static function getInstance() {
        $tracker = new Chargebee_Tracker();
        $tracker->setTrackerName('Chargebee');
        return $tracker;
    }

    protected function refundChargeback() {
        $transaction = new Pap_Common_Transaction();
        $transaction->processRefundChargeback($this->getTransactionIdFromOrderId($this->getOrderID()), Pap_Db_Transaction::TYPE_REFUND, '', $this->getOrderID(), 0, true);
    }

    public function checkStatus() {
        if (Gpf_Settings::get(Chargebee_Config::DECLINE_AFFILIATE) == Gpf::YES) {
            $this->declineAffiliate();
        }
        
        if ($this->getType() == 'payment_refunded') {
            if (Gpf_Settings::get(Chargebee_Config::TRACK_REFUND) != Gpf::YES) {
                $this->debug('Status "Refunded" retrieved, but refund handling is not enabled.');
                return false;
            }
            $this->debug('Transaction ' . $this->getOrderID() . ' will be refunded');
            try {
                $this->refundChargeback();
                $this->debug('Refund complete, ending processing.');
            } catch (Gpf_Exception $e) {
                $this->debug('Error ocured during transaction register:' . $e->getMessage());
            }
            return false;
        }
        
        if ($this->getType() == 'payment_succeeded') {
            return true;
        }
        
        return false;
    }

    private function declineAffiliate() {
        if (($this->getType() == 'payment_refunded') || ($this->getType() == 'subscription_cancelled')) {
            try {
                $affiliate = Pap_Affiliates_User::loadFromUsername($this->getEmail());
                
                if ($affiliate->getStatus() != Pap_Common_Constants::STATUS_APPROVED) {
                    $affiliate->setStatus(Pap_Common_Constants::STATUS_DECLINED);
                    $affiliate->update(array(
                            Gpf_Db_Table_Users::STATUS 
                    ));
                    $this->debug('Affiliate with username = ' . $this->getEmail() . ' has been declined after status ' . $this->getType());
                }
            } catch (Gpf_Exception $e) {
                $this->debug('Error occurred during declining the affiliate ' . $this->getEmail() . ' [status ' . $this->getType() . ' received]. Exception: ' . $e->getMessage());
            }
        }
    }

    /**
     *  @return Pap_Tracking_Request
     */
    protected function getRequestObject() {
        $input = file_get_contents('php://input');
        return json_decode($input, true);
    }

    private function computeTotalCost($request) {
        $total = $request['transaction']['amount'];
        if (Gpf_Settings::get(Chargebee_Config::SUBTRACT_TAXES) == Gpf::YES && isset($request['invoice']['tax'])) {
            $total -= $request['invoice']['tax'];
        }
        return $total / 100; // amount in cents
    }

    public function readRequestVariables() {
        $request = $this->getRequestObject();
        $this->debug('Data received: ' . print_r($request, true));
        
        $this->setType($request['event_type']);
        $request = $request['content'];
        if (!isset($request['subscription'])) {
            return;
        }
        
        if (isset($request['transaction']) && isset($request['transaction']['amount'])) {
            $this->setTotalCost($this->computeTotalCost($request));
        }
        // hack the initial payment (we do not want to track it)
        if (isset($request['invoice']['id']) && ($request['invoice']['id'] == 1)) {
            $this->setType('initial_payment');
            return;
        }
        
        $this->setSubscriptionID($request['subscription']['id']);
        $this->setProductID($request['subscription']['plan_id']);
        $this->setCurrency($request['subscription']['currency_code']);
        
        if (array_key_exists('customer', $request)) {
            $this->readRequestAffiliateVariables($request['customer']);
        }
        
        $this->setData1($this->getEmail());
    }

    public function readRequestAffiliateVariables($customer) {
        if (isset($customer['first_name'])) {
            $this->setUserFirstName($customer['first_name']);
        }
        if (isset($customer['last_name'])) {
            $this->setUserLastName($customer['last_name']);
        }
        if (isset($customer['email'])) {
            $this->setUserEmail($customer['email']);
        }
    }

    protected function processAccountIdAndVisitorId(Pap_Tracking_ActionTracker $saleTracker, $cookie) {
        parent::processAccountIdAndVisitorId($saleTracker, $cookie);
        if (Gpf_Settings::get(Chargebee_Config::APPROVE_AFFILIATE) == Gpf::YES) {
            $this->debug('Automatic approval of affiliates with sale is enabled');
            try {
                $affiliate = Pap_Affiliates_User::loadFromUsername($this->getEmail());
                if ($affiliate->getStatus() == Pap_Common_Constants::STATUS_PENDING) {
                    $affiliate->setStatus(Pap_Common_Constants::STATUS_APPROVED);
                    $affiliate->update();
                }
            } catch (Gpf_Exception $e) {
                $this->debug('Affiliate approval failed: ' . $e->getMessage());
            }
        }
    }

    public function getOrderID() {
        return $this->getSubscriptionID();
    }

    protected function getRecurringTotalCost() {
        if (Gpf_Settings::get(Chargebee_Config::RECURRING_TOTALCOST_FROM_NOTIFICATION) == Gpf::YES) {
            return $this->getTotalCost();
        }
        return null;
    }

    public function isRecurring() {
        return true;
    }

    protected function isAffiliateRegisterAllowed() {
        return (Gpf_Settings::get(Chargebee_Config::REGISTER_AFFILIATE) == Gpf::YES);
    }
}
