<?php
/**
 *   @copyright Copyright (c) 2018 Quality Unit s.r.o.
 *   @author Martin Pullmann
 *   @package PostAffiliatePro
 *   @since Version 1.0.0
 *
 *   Licensed under the Quality Unit, s.r.o. Standard End User License Agreement,
 *   Version 1.0 (the "License"); you may not use this file except in compliance
 *   with the License. You may obtain a copy of the License at
 *   http://www.postaffiliatepro.com/licenses/license
 *
 */

/**
 * @package PostAffiliatePro plugins
 */
class BigCommerceAPIv3_Config extends Gpf_Plugins_Config {
    const CONFIG_IDENTIFIER = 'BigCommerceAPIv3ConfigIdentifier';
    const API_USERNAME = 'BigCommerceAPIv3Username';
    const API_PATH = 'BigCommerceAPIv3Path';
    const API_TOKEN = 'BigCommerceAPIv3Token';
    const PER_PRODUCT = 'BigCommerceAPIv3PerProduct';
    const COUPON_TRACKING = 'BigCommerceAPIv3CouponTracking';
    const DATA1 = 'BigCommerceAPIv3ExtraData1';
    const DATA2 = 'BigCommerceAPIv3ExtraData2';
    const DATA3 = 'BigCommerceAPIv3ExtraData3';
    const DATA4 = 'BigCommerceAPIv3ExtraData4';
    const DATA5 = 'BigCommerceAPIv3ExtraData5';

    // Internal
    const WEBHOOK_ID = 'BigCommerceAPIv3WebhookID';

    protected function initFields() {
        $tracking = array(
                '0' => $this->_('order'),
                'id' => $this->_('product ID'),
                'sku' => $this->_('product SKU')
        );

        $extraData = array(
                '0' => '',
                'product_id' => $this->_('Product ID'),
                'product_name' => $this->_('Product name'),
                'sku' => $this->_('Product SKU'),
                'base_price' => $this->_('Product base price'),
                'quantity' => $this->_('Product quantity'),
                'customer_email' => $this->_('Customer email'),
                'customer_name' => $this->_('Customer name'),
                'customer_country' => $this->_('Customer country')
        );

        $this->addTextBox($this->_('Unique identifier'), self::CONFIG_IDENTIFIER, $this->_('This has to be exactly the same as you had set up in the sale tracking code which was added to your store.'));
        $this->addTextBox($this->_('API Path'), self::API_PATH, $this->_('e.g. https://api.bigcommerce.com/stores/123456/ - all your API credentials can be found in your BigCommerce admin - Advanced Settings > API Accounts > Create API Account.'));
        $this->addTextBox($this->_('Client ID'), self::API_USERNAME, $this->_('All your API credentials can be found in your BigCommerce admin - Advanced Settings > API Accounts > Create API Account.'));
        $this->addTextBox($this->_('Access Token'), self::API_TOKEN, $this->_('All your API credentials can be found in your BigCommerce admin - Advanced Settings > API Accounts > Create API Account.'));
        $this->addListBox($this->_('Track commission per'), self::PER_PRODUCT, $tracking, $this->_('If you want to track each product separately (per product tracking), choose either product ID or product SKU'));
        $this->addCheckBox($this->_('Coupon code tracking'), self::COUPON_TRACKING, $this->_('If you want to use the Coupons feature to recognize affiliate based on the coupon code used during order then this option has to be enabled'));
        $this->addListBox($this->_('Extra data 1'), self::DATA1, $extraData, $this->_('Choose a value for sale extra data'));
        $this->addListBox($this->_('Extra data 2'), self::DATA2, $extraData, $this->_('Choose a value for sale extra data'));
        $this->addListBox($this->_('Extra data 3'), self::DATA3, $extraData, $this->_('Choose a value for sale extra data'));
        $this->addListBox($this->_('Extra data 4'), self::DATA4, $extraData, $this->_('Choose a value for sale extra data'));
        $this->addListBox($this->_('Extra data 5'), self::DATA5, $extraData, $this->_('Choose a value for sale extra data'));
    }

    /**
     * @service plugin_config write
     * @param Gpf_Rpc_Params $params
     * @return Gpf_Rpc_Form
     */
    public function save(Gpf_Rpc_Params $params) {
        $form = new Gpf_Rpc_Form($params);
        $formId = $form->getFieldValue(Gpf_View_FormService::ID);
        if (!$this->isMandatoryFieldsFilled($form)) {
            return $form;
        }

        $savedIdentifiers = @unserialize(Gpf_Settings::get(self::CONFIG_IDENTIFIER));
        if (is_array($savedIdentifiers)) {
            if (isset($savedIdentifiers[$formId])) {
                unset($savedIdentifiers[$formId]);
            }
            if (in_array($form->getFieldValue(self::CONFIG_IDENTIFIER), $savedIdentifiers)) {
                $form->setFieldError(self::CONFIG_IDENTIFIER, $this->_('This identifier already exists in a different config, please choose a new one.'));
                $form->setErrorMessage($this->_('This identifier already exists in a different config, please choose a new one.'));
                return $form;
            }
        }
        $form->setInfoMessage($this->_('BigCommerce API settings saved'));

        $oldUsername = self::getSettingForId($formId, self::API_USERNAME);
        $oldToken = self::getSettingForId($formId, self::API_TOKEN);

        $hostname = str_replace('http://', 'https://', $form->getFieldValue(self::API_PATH));
        $hostname = str_replace('/v2', '', $hostname);
        $hostname = str_replace('/v3', '', $hostname);
        $this->setSettingForId($formId, self::API_PATH, $hostname);

        $this->setSettingForId($formId, self::CONFIG_IDENTIFIER, $form->getFieldValue(self::CONFIG_IDENTIFIER));
        $this->setSettingForId($formId, self::API_USERNAME, $form->getFieldValue(self::API_USERNAME));
        $this->setSettingForId($formId, self::API_TOKEN, $form->getFieldValue(self::API_TOKEN));
        $this->setSettingForId($formId, self::PER_PRODUCT, $form->getFieldValue(self::PER_PRODUCT));
        $this->setSettingForId($formId, self::COUPON_TRACKING, $form->getFieldValue(self::COUPON_TRACKING));
        $this->setSettingForId($formId, self::DATA1, $form->getFieldValue(self::DATA1));
        $this->setSettingForId($formId, self::DATA2, $form->getFieldValue(self::DATA2));
        $this->setSettingForId($formId, self::DATA3, $form->getFieldValue(self::DATA3));
        $this->setSettingForId($formId, self::DATA4, $form->getFieldValue(self::DATA4));
        $this->setSettingForId($formId, self::DATA5, $form->getFieldValue(self::DATA5));

        // set webhook
        if ($oldUsername != $form->getFieldValue(self::API_USERNAME) || $oldToken != $form->getFieldValue(self::API_TOKEN)) {
            $connection = array(
                'url' => $hostname,
                'token' => $form->getFieldValue(self::API_TOKEN),
                'clientId' => $form->getFieldValue(self::API_USERNAME)
            );

            // delete old
            if (self::getSettingForId($formId, self::WEBHOOK_ID, '0') != '') {
                try {
                    BigCommerceAPIv3_Main::api('DELETE', 'v2/hooks/'.self::getSettingForId($formId, self::WEBHOOK_ID, '0'), $connection);
                } catch (Exception $e) {
                    $form->setErrorMessage($e->getMessage());
                }
            }

            // create new
            $APIparams = array(
                    'scope' => 'store/order/statusUpdated',
                    'destination' => 'https://'.rtrim($_SERVER['HTTP_HOST'], '/').'/plugins/BigCommerceAPIv3/bigcommerce.php',
                    'is_active' => true
            );
            try {
                $response = BigCommerceAPIv3_Main::api('POST', 'v2/hooks', $connection, $APIparams);
                if (is_array($response) && isset($response['id'])) {
                    $this->setSettingForId($formId, self::WEBHOOK_ID, $response['id']);
                } else {
                    $form->setErrorMessage(print_r($response, true));
                }
            } catch (Exception $e) {
                $form->setErrorMessage($e->getMessage());
                return $form;
            }
        }
        return $form;
    }

    /**
     * @service plugin_config read
     * @param Gpf_Rpc_Params $params
     * @return Gpf_Rpc_Form
     */
    public function load(Gpf_Rpc_Params $params) {
        $form = new Gpf_Rpc_Form($params);
        $formId = $form->getFieldValue(Gpf_View_FormService::ID);
        $form->addField(self::CONFIG_IDENTIFIER, self::getSettingForId($formId, self::CONFIG_IDENTIFIER));
        $form->addField(self::API_USERNAME, self::getSettingForId($formId, self::API_USERNAME));
        $form->addField(self::API_PATH, self::getSettingForId($formId, self::API_PATH));
        $form->addField(self::API_TOKEN, self::getSettingForId($formId, self::API_TOKEN));
        $form->addField(self::PER_PRODUCT, self::getSettingForId($formId, self::PER_PRODUCT, '0'));
        $form->addField(self::COUPON_TRACKING, self::getSettingForId($formId, self::COUPON_TRACKING, Gpf::NO));
        $form->addField(self::DATA1, self::getSettingForId($formId, self::DATA1, '0'));
        $form->addField(self::DATA2, self::getSettingForId($formId, self::DATA2, '0'));
        $form->addField(self::DATA3, self::getSettingForId($formId, self::DATA3, '0'));
        $form->addField(self::DATA4, self::getSettingForId($formId, self::DATA4, '0'));
        $form->addField(self::DATA5, self::getSettingForId($formId, self::DATA5, '0'));
        return $form;
    }

    protected function getTabsCount() {
        $tabsCount = 1;
        $settingValueArray = @unserialize(Gpf_Settings::get(self::CONFIG_IDENTIFIER));
        if (is_array($settingValueArray)) {
            $arrayKeys = array_keys($settingValueArray);
            sort($arrayKeys);
            $tabsCount = end($arrayKeys) + 1;
        }
        return $tabsCount;
    }

    private function isMandatoryFieldsFilled($form) {
        $isError = false;
        if (trim($form->getFieldValue(self::API_PATH)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('API Path'));
            $form->setFieldError(self::API_PATH, $errorMessage);
            $isError = true;
        }
        if (trim($form->getFieldValue(self::API_USERNAME)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('Client ID'));
            $form->setFieldError(self::API_USERNAME, $errorMessage);
            $isError = true;
        }
        if (trim($form->getFieldValue(self::API_TOKEN)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('Access Token'));
            $form->setFieldError(self::API_TOKEN, $errorMessage);
            $isError = true;
        }
        if (trim($form->getFieldValue(self::CONFIG_IDENTIFIER)) == '') {
            $errorMessage = $this->_('%s is mandatory', $this->_('Unique identifier'));
            $form->setFieldError(self::CONFIG_IDENTIFIER, $errorMessage);
            $isError = true;
        }
        if ($isError) {
            $form->setErrorMessage($this->_('Please fill the mandatory fields.'));
            return false;
        }
        return true;
    }
}
